(ns speccy.core
  (:require [fs.core :as fs]
            [clj-biosequence.core :refer [biosequence->file]]
            [clojure.string :refer [split upper-case trim]]
            [clojure.java.io :refer [reader]]
            [clj-biosequence.indexing :as ind]
            [taoensso.nippy :as nippy])
  (:import [org.apache.commons.codec.binary Base64]))

(defprotocol Ion
  (mz [this]
    "Returns the mz as a bigdec.")
  (charge [this]
    "Returns the charge as an int, nil if no charge info.")
  (intensity [this]
    "Returns the intensity as a bigdec, nil if no intensity info."))

(def ^{:doc "Default implementation of Ion protocol."}
  default-speccy-ion
  {:mz (fn [_] nil)
   :charge (fn [_] nil)
   :intensity (fn [_] nil)})

(defprotocol Window
  (upper-limit [this]
    "Returns the upper limit of a window as a bigdec.")
  (lower-limit [this]
    "Returns the lower limit of a window as a bigdec.")
  (mz-step [this]
    "Returns the mz step of a window as a bigdec.")
  (target-mz [this]
    "Returns the target mz of a window as a bigdec."))

(defprotocol Info
  (elution-time [this]
    "Returns a vector with elution time as a bigdec and the unit."))

(defprotocol File
  (file-path [this]
    "Returns the path of the file as a java.io.File object."))

(defprotocol ID
  (accession [this]
    "Returns the accession as a string.")
  (definition [this]
    "Returns a string describing the record."))

(defprotocol Binary
  (binary-data [this]
    "Returns the binary array data.")
  (compressed? [this]
    "Returns a boolean indicating whether the record contains
    compressed data.")
  (array-length [this]
    "Returns the the length of the binary array.")
  (encoded-length [this]
    "Returns the length of the encoded length of the binary array.")
  (get-buffer [this data]
    "Returns a buffer on the binary data passed as 'data'"))

(defprotocol Spectra
  (centroid? [this]
    "Is the spectra centroided or not, returns boolean")
  (precursors [this]
    "Returns a list of precursor records.")
  (products [this]
    "Returns a list of product records.")
  (scan-list [this]
    "Returns a scan list record.")
  (binary-arrays [this]
    "Returns a list of binary array records.")
  (spot-id [this]
    "Returns the spot id as a string.")
  (mz-array [this]
    "Convenience function that returns an array of mz values. If
    record contains multiple mz binary arrays will return the first.")
  (intensity-array [this]
    "Convenience function that returns an array of intensity
    values. If record contains multiple intensity binary arrays will
    return the first.")
  (polarity [this]
    "Returns a string describing the polarity.")
  (index [this]
    "Returns the zero-based index.")
  (msn [this]
    "Returns the ms level as an integer."))

(defprotocol Precursor
  (selected-ions [this]
    "Returns selected ion records from a precursor.")
  (isolation-window [this]
    "Returns the isolation window record from a precursor.")
  (activation [this]
    "Returns the activation record from a precursor."))

(extend nil
  Precursor
  {:selected-ions (fn [this] nil)
   :isolation-window (fn [this] nil)
   :activation (fn [this] nil)})

(defprotocol Parameter
  (instrument-config [this]
    "Returns a hash of instrument config records, key is the
    accession of each config.")
  (source-files [this]
    "Returns a hash of source file records, key is the accession of
    the file.")
  (file-content [this]
    "Returns a file content record.")
  (contacts [this]
    "Returns a list of contact records.")
  (samples [this]
    "Returns a hash of sample records, keys are the accession of each
    record.")
  (software [this]
    "Returns a hash of software records, keys are the accession of
    each record.")
  (processing [this]
    "Returns a hash of processing record, keys are the accession of
    each record."))

(defprotocol Software
  (version [this]
    "Returns a version string."))

(defprotocol Chromatogram
  (time-array [this]
    "Returns time values in an array."))

(defprotocol spectraReader
  (spectra-seq [this]
    "Returns a lazy list of spectra.")
  (get-spectra [this accession]
    "Gets the spectra with the supplied accession.")
  (parameters [this]
    "Returns a parameter record.")
  (chromatogram-seq [this]
    "Returns a lazy list of chromatograms.")
  (get-chromatogram [this accession]
    "Gets the chromatogram with supplied accession."))

(defprotocol spectraFile
  (spectra-reader [this]
    "Returns a spectra reader on a spectra file."))

;;;;;;;;;;;;;;;;;;
;; binary array
;;;;;;;;;;;;;;;;;;

(defn- inflate 
  [s]
  (let [h (new java.util.zip.InflaterInputStream
               (new java.io.ByteArrayInputStream s)
               (new java.util.zip.Inflater))]
    (try
      (loop [st h
             acc nil]
        (if (= (.available h) 0)
          (byte-array (map unchecked-byte (reverse acc)))
          (recur h (cons (.read h) acc))))
      (catch Exception e
        (println e))
      (finally
        (.close h)))))

(defn decode-binary
  "Takes something implementing the Binary protocol and decodes
  it. Checks if compressed and assumes zlib compression if
  'compressed?' returns true."
  [this]
  (let [decoded (.decode (Base64.) (binary-data this))
        inflated (-> (if (compressed? this) (inflate decoded) decoded)
                     java.nio.ByteBuffer/wrap
                     (.order java.nio.ByteOrder/LITTLE_ENDIAN))
        buffer (get-buffer this inflated)]
    (loop [d buffer
           acc nil]
      (if (not (.hasRemaining d))
        (vec (reverse acc))
        (recur d (cons (.get d) acc))))))

;;;;;;;;;;;;;;;
;; indexing
;;;;;;;;;;;;;;;

(defn is-indexed?
  "Returns true if spectra file is indexed."
  [file]
  (and (fs/exists? (str (file-path file) ".idx"))
       (fs/exists? (str (file-path file) ".bin"))))

(defn delete-indexes
  "Deletes index files of a spectra file."
  [bioseq-file]
  (ind/delete-index (file-path bioseq-file)))

;;;;;;;;;;;;;;;
;; mgf
;;;;;;;;;;;;;;;

(load "mgf")
