(ns spand.core
  (:require [taoensso.sente :as sente]
            [posh.reagent :as p]
            [datascript.core :as d]
            [spand.config :refer [config]]))

(let [{:keys [chsk ch-recv send-fn state]}
      (sente/make-channel-socket! "/chsk"
       {:type :auto, :host (:authority config)})]
  (def chsk       chsk)
  (def ch-chsk    ch-recv)
  (def chsk-send! send-fn)
  (def chsk-state state))

(defmulti handler #(first (:?data %)))
(defmethod handler :default [] nil)

(sente/start-client-chsk-router! ch-chsk handler)

(defonce conn
  (let [conn (d/create-conn {})]
    (p/posh! conn)
    conn))

(defmethod handler :all/datoms
  [{[_ datoms :as ?data] :?data}]
  (let [tx-data  (map #(apply vector :db/add %) datoms)
        eids     (map first (d/datoms (d/db conn) :eavt))
        tx-clear (map #(vector :db/retractEntity %) eids)]
    (d/transact! conn tx-clear)
    (d/transact! conn tx-data)))

; Public API
; ==============================================================================

(defn q
  [query & inputs]
  (apply p/q query conn inputs))

(defn pull
  [expr eid]
  (p/pull conn expr eid))

(defn transact!
  [tx-data]
  (chsk-send! [:foo/tx tx-data]))

; TODO make a global config entity for each user and store key-values on that.
(defn get-key
  ([k default]
   (q [:find '?val '.
       :in '$ '?key '?default
       :where (if (nil? default)
                '[_ ?key ?val]
                '[(get-else $ _ ?key ?default) ?val])]
        k default))
  ([k]
   (get-key k nil)))

(defn set-key!
  [k v]
  (let [eid (if-some [datom (first (d/datoms (d/db conn) :aevt k))]
              (:e datom)
              "tempid")]
    (transact! [[:db/add eid k v]])))

; TODO make pull work in posh's q, or at least do something so that
; we're not dereffing in here.
; Also is there a way to provide the attrs as an input to q instead of building
; the where clauses dynamically? Maybe this doesn't matter for datascript.
(defn select
  [& attrs]
  (let [query (concat '[:find [?e ...] :where]
                      (for [a attrs]
                        ['?e a]))
        eids @(q query)]
    (map #(deref (pull '[*] %)) eids)))

(defn entity
  [eid]
  @(pull '[*] eid))
