(ns solita.etp.api.response
  (:require [ring.util.response :as r])
  (:import (clojure.lang ExceptionInfo)))

(defn get-response [body not-found]
  (if (nil? body)
    (r/not-found not-found)
    (r/response body)))

(defn put-response [updated not-found]
  (if (or (nil? updated) (= updated 0))
    (r/not-found not-found)
    (r/response nil)))

(defn- matches-description? [error error-description]
  (let [matcher (dissoc error-description :response)
        matched-error (select-keys error (keys matcher))]
    (= matcher matched-error)))

(defn with-exceptions
  "Convert exceptions defined in error-descriptions to error responses.
  If exception data matches error description then it is returned as a response.
  The http response status is defined in error description.
  These exceptions must not contain any sensitive data."
  ([response-fn error-descriptions]
   (try
     (response-fn)
     (catch ExceptionInfo e
       (let [error (ex-data e)
             description (first (filter (partial matches-description? error)
                                        error-descriptions))]
         (if (nil? description)
           (throw e)
           {:status  (:response description)
            :headers {}
            :body    error}))))))

(defn response-with-exceptions
  ([service-fn error-descriptions] (response-with-exceptions 200 service-fn error-descriptions))
  ([status service-fn error-descriptions]
   (with-exceptions
     (fn []
       {:status  status
        :headers {}
        :body    (service-fn)})
     error-descriptions)))

(defn created [path id]
  (r/created (str path "/" id) {:id id}))

(def forbidden {:status 403 :body "Forbidden"})

(defn file-response [body filename content-type inline? not-found]
  (if (nil? body)
    (r/not-found not-found)
    {:status 200
     :headers {"Content-Type" content-type
               "Content-Disposition:" (str (if inline? "inline" "attachment")
                                              (str "; filename=\"" filename "\""))}
     :body body}))

(defn pdf-response [body filename not-found]
  (file-response body filename "application/pdf" true not-found))

(defn xlsx-response [body filename not-found]
  (file-response body
                 filename
                 "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
                 false
                 not-found))

(defn conflict [body]
  {:status 409
   :body body})

(defn signature-response [result id]
  (cond
    (= result :already-signed)
    (conflict (str "Energiatodistus " id " is already signed"))

    (= result :already-in-signing)
    (conflict (str "Energiatodistus " id " is already in signing process"))

    (= result :not-in-signing)
    (conflict (str "Signing process for energiatodistus " id " has not been started"))

    (= result :pdf-exists)
    (conflict (str "Signed PDF exists for energiatodistus " id ". Get digest to sign again."))

    (nil? result)
    (r/not-found (str "Energiatodistus " id " does not exist"))

    (= result :ok)
    (r/response "Ok")

    :else
    (r/response result)))

(defn ->xml-response [response]
  (assoc-in response [:headers "Content-Type"] "application/xml"))

(defn bad-request [body]
  {:status 400
   :headers {}
   :body body})

(defn internal-server-error [body]
  {:status 500
   :headers {}
   :body body})
