(ns wexin.core
  (:require [clj-http.client :as http]
            [cheshire.core :as json]
            [clojure.xml :as xml])
  (:import [com.qq.weixin.mp.aes WXBizMsgCrypt])
  (:gen-class))
;;;;

(defn- parse-msg-seq [msg-xml-str]
  (xml-seq (xml/parse (java.io.ByteArrayInputStream. (.getBytes msg-xml-str "utf8")))))

(defn the-only-value-in-col [col]
  (cond
    (or (string? col) (number? col))
    col
    (coll? col)
    (the-only-value-in-col (first col))
    (nil? col)
    ""))

(defn get-value-with-tag
  [xml tag]
  (the-only-value-in-col
   (for [node (cond
                (= (type xml) "String")
                (parse-msg-seq xml)
                (seq? xml)
                xml)
         :when (= tag (:tag node))]
     (:content node))))

(defn xml-string [data]
  (clojure.string/join ""
                       (for [[k v] data]
                         (cond
                           (map? v)
                           (format "<%s>%s</%s>" (name k) (xml-string v) (name k))
                           (string? v)
                           (format "<%s><![CDATA[%s]]></%s>" (name k) v (name k))
                           :else
                           (str "<%s>%s</%s>" (name k)  v  (name k))))))
;;;;
(defn reply-msg
  [{:keys [wechat-token aes-key corpid appid reply]}]
  (let [tc (System/currentTimeMillis)]
    (.encryptMsg (WXBizMsgCrypt. wechat-token aes-key (or appid corpid)) reply tc tc)))

;;;;
(defmulti api
  "微信公众号api-多重方法"
  (fn [{:keys [type api] :as params}]
    {:type type :api api}))

;;;;基础认证接口
(defmethod api {:type :ding-yue-hao :api :base-support-access-token}
  [{:keys [appid secret]}]
  (http/get "https://api.weixin.qq.com/cgi-bin/token"
            {:query-params
             {"grant_type" "client_credential"
              "appid" appid
              "secret" secret}
             :accept :json
             :as :json}))
;;;; 订阅号 校验URL
(defmethod api {:type :ding-yue-hao :api :verify-url}
  [{:keys [wechat-token aes-key appid signature timestamp nonce echostr]}]
      ;(println wechat-token aes-key appid "\n ---" signature timestamp nonce echostr)
   ;(.verifyUrl (WXBizMsgCrypt. wechat-token aes-key appid)  signature timestamp nonce echostr)
  echostr)
;;;; 企业微信 校验URL
(defmethod api {:type :qi-ye-we-xin :api :verify-url}
  [{:keys [wechat-token aes-key corpid msg_signature timestamp nonce echostr]}]
  (.verifyUrl (WXBizMsgCrypt. wechat-token aes-key corpid)  msg_signature timestamp nonce echostr))
;;;; 订阅号 重定向到
(defmethod api {:type :ding-yue-hao :api :SSOAuth}
  [{:keys [appid redirect-uri scope]}]
  (str "https://open.weixin.qq.com/connect/oauth2/authorize?appid=" appid "&redirect_uri=" redirect-uri "&response_type=code&scope=" scope "#wechat_redirect"))
;;;; 企业微信 重定向到
(defmethod api {:type :qi-ye-we-xin :api :SSOAuth}
  [{:keys [corpid redirect-uri scope agentid]}]
  (str "https://open.weixin.qq.com/connect/oauth2/authorize?appid=" corpid "&redirect_uri=" redirect-uri "&response_type=code&scope=" scope "&agentid=" agentid "#wechat_redirect"))

(defmethod api {:type :ding-yue-hao :api :SSO-access-token}
  [{:keys [appid secret code]}]
  ;; access-token and openid
  (http/get "https://api.weixin.qq.com/sns/oauth2/access_token"
            {:query-params
             {"appid" appid
              "secret" secret
              "code" code
              "grant_type" "authorization_code"}
             :accept :json
             :as :json}))
;;;;;
(defmethod api {:type :qi-ye-we-xin :api :SSO-access-token}
  [{:keys [appid corpid secret code]}]
  (let [token
        (http/get "https://qyapi.weixin.qq.com/cgi-bin/gettoken"
                  {:query-params
                   {"corpid" (or corpid appid)
                    "corpsecret" secret}}
                  :as :json)
        ac (:body (:access_token token))
        r
        (http/get "https://qyapi.weixin.qq.com/cgi-bin/user/getuserinfo"
                  {:query-params
                   {"access_token" ac "code" code}
                   :accept :json :as :json})]
    ;; return ticket in body
    r))

(defmethod api {:type :ding-yue-hao :api :SSO-refresh-token}
  [{:keys [appid refresh-token]}]
  (http/get "https://api.weixin.qq.com/sns/oauth2/refresh_token"
            {:query-params
             {"appid" appid
              "grant_type" "refresh_token"
              "refresh_token" refresh-token}
             :accept :json
             :as :json}))

;;;; 拉取用户信息
(defmethod api {:type :ding-yue-hao :api :SSO-user-info}
  [{:keys [access_token openid]}]
  (http/get "https://api.weixin.qq.com/sns/userinfo"
            {:query-params
             {"access_token" access_token
              "openid" openid
              "lang" "zh_CN"}
             :as :json
             :accept :json}))
(defmethod api {:type :qi-ye-we-xin :api :SSO-user-info}
  [{:keys [access_token ticket]}]
  (http/post (str "https://qyapi.weixin.qq.com/cgi-bin/user/getuserdetail?access_token=" access_token)
             {:body (json/generate-string {:user_ticket ticket})
              :content-type :json :accept :json
              :as :json}))

;;用户相关     
(defmethod api {:type :qi-ye-we-xin :api :users-user-info}
  [{:keys [userid access_token]}]
  (http/get "https://qyapi.weixin.qq.com/cgi-bin/user/get"
            {:query-params
             {"access_token" access_token
              "userid" userid}
             :as :json
             :accept :json}))

;;;;发送消息接口 
;;;;see @ https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140547
(defmethod api {:type :ding-yue-hao :api :send-msg}
  [{:keys [access_token msg]}]
  (http/post (str "https://api.weixin.qq.com/cgi-bin/message/custom/send?access_token=" access_token)
             {:body msg
              :content-type :json :accept :json}))
;;       


(defmethod api {:type :qi-ye-we-xin :api :send-msg}
  [{:keys [access_token msg]}]
  (http/post (str "https://qyapi.weixin.qq.com/cgi-bin/message/send?access_token=" access_token)
             {:body msg
              :content-type :json :accept :json}))
;;;;;;

(defmulti msg (fn [x] (:MsgType x)))

(defmethod msg :default [params]
  (reply-msg (xml-string params)))

(defmethod msg :text
  [{:keys [ToUserName FromUserName CreateTime MsgType Content] :as params}]
  (reply-msg (xml-string params)))

(defmethod msg :image
  [params]
  (let [{:keys [ToUserName FromUserName CreateTime MsgType Content Image]} params
        {:keys [MediaId]} Image]
    (reply-msg (xml-string params))))

