(ns {{sanitized}}.util
      (:refer-clojure :exclude [max min get + - * / quot mod == rem contains? get-in < <= > >=
                                boolean re-find and or count str nth rand nil? empty? not])
      (:require
        [{{sanitized}}.internal :as internal]
        [cheshire.core :as json]
        [aleph.http :as http])
      (:import (java.time Instant)))


(def root-auth (Long/valueOf (slurp "src/{{sanitized}}/rootAuth.txt")))

(defn resolve-two-tuple
  [subject]
  (-> (internal/query {:select "?sid"
                       :where  [["?sid" (first subject) (second subject)]]}) first))

(defn resolve-cname
  [_id]
  (if    (number? _id)
    (let [collection-sid (-> (bit-shift-right _id 44)
                             (bit-or 17592186044416))
          cname-query {:select "?cName"
                       :where [[collection-sid "_collection/name" "?cName"]]}
          res       (-> (internal/query cname-query) first)]
      res)
    (clojure.core/re-find #"^[a-zA-Z0-9_][a-zA-Z0-9\.\-_]{0,254}" _id)))

(defn get-sid
  [subject]
  (cond
    (number? subject)
    subject

    (string? subject)
    nil

    (clojure.core/and (vector? subject) (= 2 (clojure.core/count subject)))
    (resolve-two-tuple subject)))

(defn get-pid
  [predicate-name]
  (if (clojure.core/nil? predicate-name)
    nil
    (clojure.core/or (-> (internal/query {:select "?id" :where  [["?id" "_predicate/name" predicate-name]]})
                         first) (throw (ex-info "Invalid predicate provided."
                                                {:status 400
                                                 :error :db/invalid-txn})))))

(defn keyword->str
  "Converts a keyword to string. Can safely be called on a
  string which will return itself."
  [k]
  (cond
    (keyword? k) (subs (clojure.core/str k) 1)
    (string? k) k
    :else (throw (ex-info (clojure.core/str "Cannot convert type " (type k) " to string: " (pr-str k))
                          {:status 500 :error :db/unexpected-error}))))

(defn namespace-predicate
  [collection predicate]
  (if (.contains predicate "/")
    predicate
    (clojure.core/str collection "/" predicate)))

(defn get-predicate+tx-specs
  [predicate]
  (let [res (internal/query {:selectOne {"?predicate" [{"spec" ["*"]} {"txSpec" ["*"]}]}
                             :where [["?predicate" "_predicate/name" predicate]]})
        pred-map (apply merge res)]
    (if (clojure.core/empty? pred-map)
      nil (assoc pred-map :name predicate))))

(defn replace-fn
  [funStr matches replacement shift]
  (reduce (fn [funCnt match]
            (let [[fun last-idx] funCnt
                  startMatch (string/index-of fun match last-idx)
                  length     (clojure.core/count match)
                  endMatch   (clojure.core/- (clojure.core/+ startMatch length) shift)
                  funStr'    (clojure.core/str (subs fun 0 endMatch) replacement (subs fun endMatch))]
              [funStr' endMatch]))
          [funStr 0] matches))

(defn filter-vals
  "Filters map k/v pairs dropping any where predicate applied to value is false."
  [pred m]
  (reduce-kv (fn [m k v] (if (pred v) (assoc m k v) m)) {} m))

(defn without-nils
  "Remove all keys from a map that have nil or empty collection values."
  [m]
  (filter-vals #(if (coll? %) (not-empty %) (some? %)) m))

(defn insert-ctx-param
  [funStr]
  (let [matches         (re-seq #"\([a-zA-Z-\?\<\>=\_]+ " funStr)
        [funStr']       (replace-fn funStr matches "ctx " 0)
        matches'        (re-seq #"\([a-zA-Z-\?\<\>=\_]+\)" funStr')
        [funStr'']      (replace-fn funStr' matches' " ctx" 1)] funStr''))

(defn gen-flakes
  [txn]
  (let [{:keys [ip network db]} (internal/config)]
    (try (-> @(http/post (clojure.core/str ip "/fdb/" network "/" db "/gen-flakes")
                         {:content-type    :json
                          :request-timeout 5000                 ;; 5 seconds
                          :body            (json/encode txn)})
             :body
             internal/json-read
             :flakes)
         (catch Exception e (clojure.core/str "Error in gen-flakes: " (-> (ex-data e)
                                                                          :body
                                                                          internal/json-read))))))

(defn gen-ctxs
  [flakes opts]
  (let [{:keys [type auth_id predicate collection]} opts
        instant (Instant/now)
        pid     (get-pid predicate)
        _ (when (clojure.core/and (clojure.core/nil? pid)  (#{"_predicate/spec"} type))
            (throw (ex-info "When testing a _predicate/spec, you must provide the predicate name."
                            {:status 400
                             :error :db/invalid-txn})))
        _ (when (clojure.core/and (clojure.core/nil? collection)  (#{"_collection/spec"} type))
            (throw (ex-info "When testing a _collection/spec, you must provide the collection name."
                            {:status 400
                             :error :db/invalid-txn})))
        test-iterators (condp = type
                         "_predicate/spec"
                         (let [flakes'  (filterv #(clojure.core/and
                                                    (= true (clojure.core/nth % 4))
                                                    (= pid (second %))) flakes)]
                           (map (fn [flake]
                                  {:sid (first flake)
                                   :flakes [flake]
                                   :o (clojure.core/nth flake 2)}) flakes'))

                         "_predicate/txSpec"
                         (let [flakes'  (filterv #(= pid (second %)) flakes)
                               sid      (-> flakes' first first)]
                           [{:sid sid
                             :flakes flakes'}])

                         "_collection/spec"
                         (let [flakes'  (filterv #(= collection (resolve-cname (first %))) flakes)]
                           (map (fn [flake]
                                  {:sid (first flake)
                                   :flakes [flake]}) flakes')))
        ctx-base (-> {:instant     instant
                      :query-with  flakes
                      :auth_id     (clojure.core/or auth_id root-auth)
                      :pid         pid
                      :print-stack true
                      :state       (atom {:stack   []
                                          :credits 10000000
                                          :spent   0})} without-nils)]
    (map #(merge ctx-base %) test-iterators)))


(defn find-all-applicable-rules-txi
  [tx authId]
  ;; Find collection spec
  ;; Find predicate and predicate tx spec
  ;; Find all auth -> rules
  (let [_id         (:_id tx)
        authId      (clojure.core/or authId root-auth)
        collection  (if (clojure.core/and (vector? _id) (= 2 (clojure.core/count _id)))
                      (->> (resolve-two-tuple _id)
                           resolve-cname)
                      (resolve-cname _id))
        collection-spec (internal/query {:selectOne {"?collection" [{"spec" ["*"]}]}
                                         :where [["?collection" "_collection/name" collection]]})
        all-predicates (->> (dissoc tx :_id) keys
                            (mapv keyword->str) (map #(namespace-predicate collection %)))
        predicate+tx-specs (reduce (fn [acc pred]
                                     (if-let [pred+tx-spec (get-predicate+tx-specs pred)]
                                       (conj acc pred+tx-spec) acc)) [] all-predicates)
        auth-roles    (-> (internal/query {:selectOne [{:_auth/roles [{"rules" ["*"]}]}]
                                           :from authId}) :_auth/roles)]
    {:_collection {:name collection
                   :spec (:_collection/spec collection-spec)}
     :_predicate predicate+tx-specs
     :_auth {:_id authId
             :rules auth-roles}}))
