(ns {{sanitized}}.internal
  (:refer-clojure :exclude [max min get inc dec + - * / == quot mod rem contains? get-in < <= > >=
                            boolean re-find and or count str nth rand nil? empty? not])
  (:require [clojure.tools.reader.edn :as edn]
    [cheshire.core :as json]
    [aleph.http :as http]
    [clojure.tools.logging :as log]
    [byte-streams :as bs])
  (:import (java.time Instant)))

(declare str)

(defn json-read
  [x]
  (if (string? x)
    (json/decode x true)
    (json/decode-stream (bs/to-reader x) true)))

(defn config
  []
  (-> (slurp "flureeconfig.json")
      json-read))

(defn query-with
  [query-map flakes]
  (let [{:keys [ip network db]} (config)]
    (-> @(http/post (clojure.core/str ip "/fdb/" network "/" db "/query-with")
                    {:content-type    :json
                     :request-timeout 5000                 ;; 5 seconds
                     :body            (json/encode {:query query-map :flakes flakes})})
        :body
        json-read)))

(defn query
  [query-map]
  (let [{:keys [ip network db]} (config)]
    (-> @(http/post (clojure.core/str ip "/fdb/" network "/" db "/query")
                    {:content-type    :json
                     :request-timeout 5000                 ;; 5 seconds
                     :body            (json/encode query-map)})
        :body
        json-read)))

(defn filter-vals
  "Filters map k/v pairs dropping any where predicate applied to value is false."
  [pred m]
  (reduce-kv (fn [m k v] (if (pred v) (assoc m k v) m)) {} m))

(defn without-nils
  "Remove all keys from a map that have nil or empty collection values."
  [m]
  (filter-vals #(if (coll? %) (not-empty %) (some? %)) m))

(defn parse-select-map
  [param-str]
  (let [parsed-param (if (string? param-str) (edn/read-string param-str)  param-str)]
    (cond
      (map? parsed-param)
      (let [key         (first (keys parsed-param))
            key'        (if (string? key) key
                                          (if (clojure.core/nil? (namespace key))
                                            (name key)
                                            (clojure.core/str (namespace key) "/" (name key))))
            value (first (vals parsed-param))
            value' (parse-select-map value)
            value'' (if (coll? value')
                      (into [] value')
                      value')]
        (assoc {} key' value''))

      (string? parsed-param)
      parsed-param

      (clojure.core/or (symbol? parsed-param) (var? parsed-param))
      (if (clojure.core/nil? (namespace parsed-param))
        (name parsed-param)
        (clojure.core/str (namespace parsed-param) "/" (name parsed-param)))

      (vector? parsed-param)
      (mapv parse-select-map parsed-param)

      :else
      (throw (ex-info (clojure.core/str "The query path is not properly formatted: " parsed-param)
                      {:status 400
                       :error  :db/invalid-fn})))))
(defn function-error
  [e function-name & args]
  (throw (ex-info (clojure.core/str "Error in database function: " function-name ": " (if (clojure.core/nil? (.getMessage e)) (.getClass e) (.getMessage e))
                                    ". Provided: " (if (coll? args) (clojure.string/join " " args) args))
                  {:status 400
                   :error  :db/invalid-fn})))

(defn boolean
  "Coerce to boolean. Everything except `false' and `nil' is true in boolean context."
  [x]
  (try (clojure.core/boolean x)
       (catch Exception e (function-error e "boolean" x))))

(defn nil?
  [arg]
  (try (clojure.core/nil? arg)
       (catch Exception e (function-error e "nil?" arg))))

(defn not
  [arg]
  (try (clojure.core/not arg)
       (catch Exception e (function-error e "not" arg))))

(defn empty?
  [arg]
  (try (clojure.core/or (clojure.core/empty? arg) (= #{nil} arg))
       (catch Exception e (function-error e "empty?" arg))))

(defn if-else
  "Like clojure.core/if"
  [test true-res false-res]
  (try (cond test
             true-res

             :else
             false-res)
       (catch Exception e (function-error e "if-else" test true-res false-res))))

(defn and
  "Returns true if all true"
  [& args]
  (try
    (let [coerced-coll (map boolean args)]
      (if (nil? args)
        false
        (boolean (every? true? coerced-coll))))
    (catch Exception e (function-error e "and" args))))

(defn or
  "Returns true if any true"
  [& args]
  (try
    (let [coerced-coll (map boolean args)]
      (if (nil? args)
        false
        (boolean (some true? coerced-coll))))
    (catch Exception e (function-error e "or" args))))

(defn count
  "Returns the number of items in the collection. (count nil) returns 0.  Also works on strings, arrays, and Java Collections and Maps"
  [coll]
  (try (clojure.core/count coll)
       (catch Exception e (function-error e "count" coll))))

(defn str
  "Like clojure.core/str"
  [& args]
  (try (apply clojure.core/str args)
       (catch Exception e (function-error e "str" args))))

(defn lower-case
  "Like clojure.core/lower-case"
  [str]
  (try (clojure.string/lower-case str)
       (catch Exception e (function-error e "lower-case" str))))

(defn upper-case
  "Like clojure.core/upper-case"
  [str]
  (try (clojure.string/upper-case str)
       (catch Exception e (function-error e "upper-case" str))))

(defn max
  "Like clojure.core/max, but applies max on a sequence"
  [& args]
  (try (apply clojure.core/max (remove nil? args))
       (catch Exception e (function-error e "max" args))))

(defn min
  "Like clojure.core/min, but applies min on a sequence"
  [& args]
  (try (apply clojure.core/min (remove nil? args))
       (catch Exception e (function-error e "min" args))))

(defn >
  "Like clojure.core/>, but applies > on a sequence"
  [& args]
  (try (apply clojure.core/> args)
       (catch Exception e (function-error e ">" args))))

(defn >=
  "Like clojure.core/>=, but applies > on a sequence"
  [& args]
  (try (apply clojure.core/>= args)
       (catch Exception e (function-error e ">=" args))))

(defn <
  "Like clojure.core/>, but applies < on a sequence"
  [& args]
  (try (apply clojure.core/< args)
       (catch Exception e (function-error e "<" args))))

(defn <=
  "Like clojure.core/>, but applies < on a sequence"
  [& args]
  (try (apply clojure.core/<= args)
       (catch Exception e (function-error e "<=" args))))

(defn inc
  "Increments by 1. nil is treated as zero."
  [n]
  (try (if (nil? n)
         1
         (clojure.core/inc n))
       (catch Exception e (function-error e "inc" n))))

(defn dec
  "Decrements by 1. nil is treated as zero."
  [n]
  (try
    (if (nil? n)
      -1
      (clojure.core/dec n))
    (catch Exception e (function-error e "dec" n))))

(defn get
  [m k]
  (try
    (clojure.core/or (clojure.core/get m k) (clojure.core/get m (keyword k)))
    (catch Exception e (function-error e "get" m k))))

(defn now
  "Returns current epoch milliseconds."
  []
  (try
    (-> (Instant/now)
        (.toEpochMilli))
    (catch Exception e (function-error e "now"))))

(defn +
  "Returns sum of all arguments in a sequence."
  [& args]
  (try
    (apply clojure.core/+ args)
    (catch Exception e (function-error e "+" args))))

(defn -
  "Returns difference of all the numbers in the sequence with the first number as the minuend."
  [& args]
  (try
    (apply clojure.core/- args)
    (catch Exception e (function-error e "-" args))))

(defn *
  "Returns product of all the numbers in the sequence."
  [& args]
  (try (if (clojure.core/or (nil? args) (empty? args))
         1
         (apply clojure.core/* args))
       (catch Exception e (function-error e "*" args))))

(defn /
  "If no denominators are supplied, returns 1/numerator, else returns numerator divided by all of the denominators. Takes a sequence"
  [& args]
  (try (if (nil? args)
         (throw (ex-info (clojure.core/str "Function / takes at least one argument")
                         {:status 400
                          :error  :db/invalid-fn}))
         (apply clojure.core// args))
       (catch Exception e (function-error e "/" args))))

(defn quot
  "Quot[ient] of dividing numerator by denominator."
  [n d]
  (try
    (clojure.core/quot n d)
    (catch Exception e (function-error e "quot" n d))))

(defn mod
  "Modulus of num and div. Truncates toward negative infinity."
  [n d]
  (try (clojure.core/mod n d)
       (catch Exception e (function-error e "mod" n d))))

(defn rem
  "Remainder of dividing numerator by denominator."
  [n d]
  (try
    (clojure.core/rem n d)
    (catch Exception e (function-error e "rem" n d))))

(defn ceil
  "Returns the ceiling of a number, as integer."
  [num]
  (try
    (int (Math/ceil num))
    (catch Exception e (function-error e "ceil" num))))

(defn floor
  "Returns the floor of a number, as integer."
  [num]
  (try
    (int (Math/floor num))
    (catch Exception e (function-error e "floor" num))))

(defn get-all
  "Follows an subject down the provided path and returns a set of all matching subjects."
  [start-subject path]
  (try (loop [[pred & r] path
              subjects #{start-subject}]
         (let [next-subjects (reduce (fn [acc subject]
                                       (let [sub-subjects (if (vector? subject)
                                                            (mapv #(get % pred) subject)
                                                            (get subject pred))]
                                         (if
                                           (clojure.core/or (vector? sub-subjects) (set? sub-subjects))
                                           ;; multi-cardinality, combine
                                           (into acc sub-subjects)

                                           ;; single-cardinality - conj
                                           (conj acc sub-subjects))))
                                     #{} subjects)]
           (if (clojure.core/and r (not-empty next-subjects))
             (recur r next-subjects)
             next-subjects)))
       (catch Exception e (function-error e "get-all" start-subject path))))

(defn get-in
  "Returns the value in a nested structure"
  [m ks]
  (try
    (get-all m ks)
    (catch Exception e (function-error e "get-in" m ks))))

(defn contains?
  "Returns true if key is present."
  [coll key]
  (try
    (clojure.core/contains? coll key)
    (catch Exception e (function-error e "contains?" coll key))))

(defn nth
  "Returns true if key is present."
  [coll key]
  (try
    (let [coll' (if (set? coll)
                  (into [] coll) coll)]
      (clojure.core/nth coll' key))
    (catch Exception e (function-error e "nth" coll key))))

(defn ==
  "Return true if arguments in sequence equal each other."
  [& args]
  (try
    (apply = args)
    (catch Exception e (function-error e "==" args))))

(defn re-find
  "Returns the next regex match, if any, of string to pattern, using java.util.regex.Matcher.find().  Uses re-groups to return the groups."
  [pattern string]
  (try (clojure.core/re-find (re-pattern pattern) string)
       (catch Exception e (function-error e "re-find" pattern string))))

(defn ?pO
  [?ctx]
  (let [sid (if (:sid ?ctx)
              (:sid ?ctx)
              (get (:s ?ctx) :_id))]
    (if (string? sid)
      [nil 0]
      (try (let [predNameQ {:select ["_predicate/name"]
                            :from   (:pid ?ctx)}
                 res      (query predNameQ)
                 predName (clojure.core/get res "_predicate/name")
                 pOQuery {:select [predName]
                          :from   sid}
                 res2 (if sid (query pOQuery) [nil nil])
                 pO (get res2 predName)]
             [pO 0])
           (catch Exception e (function-error e "?pO" "Context Object"))))))

(defn max-pred-val
  [pred-name flakes]
  (try
    (let [res (query-with {:select "?o" :where [[nil pred-name "?o"]]} flakes)]
      [(apply max res) 0])
    (catch Exception e (function-error e "max-pred-val" pred-name))))

(defn valid-email?
  [email]
  (try
    (let [pattern #"[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*@(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?"]
      (clojure.core/boolean (clojure.core/and (string? email) (re-matches pattern email))))
    (catch Exception e (function-error e "valid-email?" email))))

(defn objT
  "Given an array of flakes, returns the sum of the objects of the true flakes"
  [flakes]
  (try
    (let [trueF (filterv #(true? (clojure.core/nth % 4)) flakes)
          objs (map #(clojure.core/nth % 2) trueF)
          sum (reduce clojure.core/+ objs)]
      sum)
    (catch Exception e (function-error e "objT" flakes))))

(defn objF
  "Given an array of flakes, returns the sum of the objects of the false flakes"
  [flakes]
  (try
    (let [falseF (filterv #(false? (clojure.core/nth % 4)) flakes)
          objs (map #(clojure.core/nth % 2) falseF)
          sum (reduce clojure.core/+ objs)]
      sum)
    (catch Exception e (function-error e "objF" flakes))))

(defn rand
  [instant max']
  (try
    (let [base (.nextDouble (java.util.Random. instant))
          num (int (Math/floor (* base max')))] num)
    (catch Exception e (function-error e "rand" instant max'))))

(defn between?
  [min max x]
  (clojure.core/and (<= x max) (>= x min)))

