(ns {{sanitized}}.core
  (:refer-clojure :exclude [max min get + - * / quot mod == rem contains? get-in < <= > >=
                            boolean re-find and or count str nth rand nil? empty? not])
  (:require
    [{{sanitized}}.internal :as internal]
    [{{sanitized}}.custom_functions :refer :all]
    [{{sanitized}}.temp_custom_functions :refer :all]
    [{{sanitized}}.util :refer :all]
    [clojure.string :as string]
    [cheshire.core :as json]
    [clojure.tools.logging :as log]
    [aleph.http :as http])
  (:import (java.time Instant)))

(defn find-all-applicable-rules
  "This function takes a transaction, and returns the list of rules that apply to it, including any _predicate/spec, _predicate/txSpec, _collection/spec, and auth-specific rules.
  If you don't provide an auth record, the function uses a root-auth, otherwise it uses your provided auth record."
  ([tx]
   (find-all-applicable-rules tx root-auth))
  ([tx auth]
   (map #(find-all-applicable-rules-txi % auth) tx)))

(defn test-fn
  "test-fn allows you to test any function (from either your temp functions or your database's custom functions) as a _predicate/spec, _predicate/txSpec, or _collection/spec.

  This function takes a transaction, a function, and opts.
  The function should just be the name of a function in temp_custom_functions or custom_functions (make sure your REPL has loaded the most recent version of these namespaces).

  The opts object should have the following items:

  :type (required) - \"_predicate/spec\", \"_predicate/txSpec\", or  \"_collection/spec\"
  :predicate - The predicate name, i.e. \"person/handle\". This is required if your type is \"_predicate/spec\" or \"_predicate/txSpec\".
  :collection - The collection name, i.e. \"person\". This is required if your type is \"_collection/spec\".
  :auth_id - Any existing auth id to test the function with. If not specified, will use an auth record with root permissions.
  :params - If the function takes any params, you can specify a vector of params

  For example, if you have a function, isPositive that checks whether a value is positive. You might want to test that function as if it is attached to the person/age _predicate/spec. You would issue the following function call:

  (test-fn [{\"_id\": \"person\", \"age\": 45}] isPositive {:type \"_predicate/spec\" :predicate \"person/age\"})

  Note: this function does not actually issue the transaction to the database. Your database will NOT be updated based on calling this function."
  [txn spec opts]
  (let [{:keys [auth_id params]} opts
        auth      (clojure.core/or auth_id root-auth)
        opts      (assoc opts :auth_id auth)
        flakes    (gen-flakes txn)
        ctxs      (gen-ctxs flakes opts)
        param-sets (map #(concat [%] params) ctxs)]
    (map #(apply spec %) param-sets)))

(defn test-txn
  "This function takes a transaction. If the function succeeds, the function will return a list of resulting flakes.

  Note, you cannot use an function in the transaction that have not been added to the database. For example, if the function add20 is in temp_custom_functions, a transaction like the below WILL FAIL.

  (test-txn [{:_id \"person\", :age \"#(add10 50)\"}])

  Note: this function does not actually issue the transaction to the database. Your database will NOT be updated based on calling this function."
  [txn]
  (gen-flakes txn))

(defn blank-ctx
  "Creates a blank ctx object. You can use this if you have a function that does not depend on any context information, for example (add10 (blank-ctx) 5).
   You may also want to use this function if you feel confident editing a ctx object on your own. "
  []
  {:instant     (Instant/now)
   :print-stack true
   :state       (atom {:stack   []
                       :credits 10000000
                       :spent   0})})

(comment
  ; TEST YOUR FUNCTIONS BELOW


  )