(ns {{sanitized}}.fns
  (:refer-clojure :exclude [max min get inc dec + - * / quot mod == rem contains? get-in < <= > >=
                            boolean re-find and or count str nth rand nil? hash-set empty? not])
  (:require [{{sanitized}}.internal :as fdb]
    [cheshire.core :as cheshire]))

(defn stack
  "Returns the current stack."
  [?ctx]
  (-> @(:state ?ctx)
      :stack))

(defn add-stack
  "Adds an entry to the current stack."
  [?ctx entry]
  (let [[_ cost] entry
        _       (if (:print-stack ?ctx) (println entry))]
    (swap! (:state ?ctx) (fn [s]
                           (assoc s :stack (conj (:stack s) entry)
                                    :credits (fdb/-  (:credits s) cost)
                                    :spent (fdb/+ (:spent s) cost))))))
(defn raise
  "Throws an exception with the provided message."
  [?ctx msg]
  (throw (ex-info msg
                  {:status 400
                   :error  :db/invalid-fn
                   :stack  (stack ?ctx)})))

(defn nth
  {:doc  "Returns the nth item in a collection"
   :fdb/spec nil
   :fdb/cost "9 + count of objects in collection"}
  [?ctx coll key]
  (let [res     (fdb/nth coll key)
        cost    (clojure.core/+ 9 (clojure.core/count coll))
        entry   [{:function "nth" :arguments [coll key] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn if-else
  {:doc      "Evaluates test."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx test t f]
  (let [res     (fdb/if-else test t f)
        entry   [{:function "if-else" :arguments [test t f] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn nil?
  {:doc      "True if nil, else false."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx arg]
  (let [res     (fdb/nil? arg)
        entry   [{:function "nil?" :arguments [arg] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn not
  {:doc      "Takes a boolean, true returns false, false returns true."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx arg]
  (let [res     (fdb/not arg)
        entry   [{:function "not?" :arguments [arg] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn empty?
  {:doc      "True if empty or #{nil}, else false."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx arg]
  (let [res     (fdb/empty? arg)
        entry   [{:function "empty?" :arguments [arg] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn str
  {:doc      "Concatenates all in sequence."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx & args]
  (let [res     (apply fdb/str args)
        entry   [{:function "str" :arguments [args] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn lower-case
  {:doc      "Makes string lower case"
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx str]
  (let [res     (fdb/lower-case str)
        entry   [{:function "lower-case" :arguments [str] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn upper-case
  {:doc      "Makes string upper-case"
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx str]
  (let [res     (fdb/upper-case str)
        entry   [{:function "upper-case" :arguments [str] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn db
  {:doc      "Gets current database."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (let [res     (:db ?ctx)
        entry   [{:function "db" :arguments "?ctx" :result (clojure.core/str "DB with dbid: " (:dbid res))} 10]
        _       (add-stack ?ctx entry)] res))

(defn ?sid
  {:doc      "Gets current subject id"
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (cond
    (:sid ?ctx)
    (let [res     (:sid ?ctx)
          entry   [{:function "?sid" :arguments "?ctx" :result res} 10]
          _       (add-stack ?ctx entry)] res)

    (clojure.core/and (:s ?ctx) (not (string? (clojure.core/get-in ?ctx [:s :_id]))))
    (let [res     (clojure.core/get-in ?ctx [:s :_id])
          entry   [{:function "?sid" :arguments "?ctx" :result res} 10]
          _       (add-stack ?ctx entry)] res)

    :else
    (raise ?ctx "Cannot access ?sid from this function interface")))

(defn ?pid
  {:doc      "Gets current predicate id"
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (cond
    (:pid ?ctx)
    (let [res     (:pid ?ctx)
          entry   [{:function "?pid" :arguments "?ctx" :result res} 10]
          _       (add-stack ?ctx entry)] res)
    :else
    (raise ?ctx "Cannot access ?pid from this function interface")))

(defn and
  {:doc  "Returns true if all in a sequence are true, else returns false"
   :fdb/spec nil
   :fdb/cost "Count of objects in and"}
  [?ctx & args]
  (let [res     (apply fdb/and args)
        cost    (clojure.core/count [args])
        entry   [{:function "and" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn or
  {:doc  "Returns true if any in the sequence are true, else returns false"
   :fdb/spec nil
   :fdb/cost "Count of objects in or"}
  [?ctx & args]
  (let [res     (apply fdb/or args)
        cost    (clojure.core/count [args])
        entry   [{:function "or" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn count
  {:doc  "Returns the number of items in the collection. (count nil) returns 0.  Also works on strings, arrays, and Java Collections and Maps"
   :fdb/spec nil
   :fdb/cost "9 + count of objects in count"}
  [?ctx coll]
  (let [res     (clojure.core/count (remove clojure.core/nil? coll))
        cost    (clojure.core/+ 9 res)
        entry   [{:function "count" :arguments coll :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn get
  {:doc      "Gets a value from an subject."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx subject pred]
  (let [res     (fdb/get subject pred)
        entry   [{:function "get" :arguments [subject pred] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn ?o
  {:doc      "Gets the object of an predicate from the current subject."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (if (:o ?ctx)
    (let [res     (:o ?ctx)
          entry   [{:function "?o" :arguments "?ctx" :result res} 10]
          _       (add-stack ?ctx entry)] res)
    (raise ?ctx "Cannot access ?o from this function interface")))

(defn ?pO
  {:doc      "Gets the most recent object of an predicate, as of the previous block"
   :fdb/spec nil
   :fdb/cost "10 plus fuel cost"}
  [?ctx]
  (if (clojure.core/and (clojure.core/or (:sid ?ctx) (:s ?ctx)) (:pid ?ctx))
    (let [[res fuel]    (fdb/?pO ?ctx)
          entry         [{:function "?pO" :arguments "?ctx" :result res} (clojure.core/+ 10 fuel)]
          _             (add-stack ?ctx entry)] res)
    (raise ?ctx "Cannot access ?pO from this function interface")))

(defn get-all
  {:doc      "Follows an subject down the provided path and returns a set of all matching subjects."
   :fdb/spec nil
   :fdb/cost "9 + length of path"}
  [?ctx subject path]
  (let [subject' (if (vector? subject)
                   (if (= 1 (clojure.core/count subject))
                     (first subject)
                     subject)
                   subject)
        res   (fdb/get-all subject' path)
        cost (clojure.core/+ 9  (clojure.core/count path))
        entry   [{:function "get-all" :arguments [subject path] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn get-in
  [?ctx subject path]
  {:doc      "Returns the value of a nested structure"
   :fdb/spec nil
   :fdb/cost "Length of path"}
  [?ctx subject path]
  (let [res     (fdb/get-in subject path)
        cost    (clojure.core/count path)
        entry   [{:function "get-in" :arguments [subject path] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn contains?
  [?ctx coll key]
  {:doc      "Returns true if key is present."
   :fdb/spec nil
   :fdb/cost 10}
  (let [res     (fdb/contains? coll key)
        entry   [{:function "contains?" :arguments [coll key] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn ==
  {:doc  "Return true if arguments in sequence equal each other."
   :fdb/spec nil
   :fdb/cost "9 + count of objects in =="}
  [?ctx & args]
  (let [res     (apply fdb/== args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "==" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn >
  {:doc      "Returns non-nil if nums are in monotonically decreasing order, otherwise false."
   :fdb/spec nil
   :fdb/cost  "9 + count of numbers in >"}
  [?ctx & args]
  (let [res     (apply fdb/> args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function ">" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn <
  {:doc      "Returns non-nil if nums are in monotonically increasing order, otherwise false."
   :fdb/spec nil
   :fdb/cost "9 + count of numbers in <"}
  [?ctx & args]
  (let [res     (apply fdb/< args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "<" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn <=
  {:doc      "Returns non-nil if nums are in monotonically non-decreasing order,\notherwise false."
   :fdb/spec nil
   :fdb/cost "9 + count of numbers in <="}
  [?ctx & args]
  (let [res     (apply fdb/<= args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "<=" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn >=
  {:doc      "Returns non-nil if nums are in monotonically non-increasing order,\notherwise false."
   :fdb/spec nil
   :fdb/cost "9 + count of numbers in <="}
  [?ctx & args]
  (let [res     (apply fdb/>= args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function ">=" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn max
  {:doc      "Gets max value from a sequence."
   :fdb/spec nil
   :fdb/cost "9 + count of numbers in max"}
  [?ctx & args]
  (let [res     (apply fdb/max args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "max" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn min
  {:doc      "Gets min value from a sequence."
   :fdb/spec nil
   :fdb/cost "Count of numbers in min"}
  [?ctx & args]
  (let [res     (apply fdb/min args)
        cost    (clojure.core/count [args])
        entry   [{:function "min" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn query
  {:doc      "Executes a query"
   :fdb/spec nil
   :fdb/cost "Fuel required for query"}
  [?ctx query-string]
  (let [flakes        (:query-with ?ctx)
        query-map     (-> (cheshire/parse-string query-string)
                          clojure.walk/keywordize-keys)
        res           (fdb/query-with query-map flakes)
        entry          [{:function "query" :arguments [query-string] :result res} 0]
        _               (add-stack ?ctx entry)] res))

(defn max-pred-val
  {:doc      "Finds the maximum predicate value."
   :fdb/spec nil
   :fdb/cost "10, plus fuel cost."}
  [?ctx pred-name]
  (let [flakes        (:query-with ?ctx)
        [res fuel]    (fdb/max-pred-val pred-name flakes)
        entry         [{:function "max-pred-val" :arguments pred-name :result res} (clojure.core/+ fuel 10)]
        _             (add-stack ?ctx entry)] res))

(defn inc
  {:doc      "Increments any number (or nil/null) by 1."
   :fdb/spec nil
   :fdb/cost 10}
  ([?ctx] (inc ?ctx 0))
  ([?ctx n]
   (let [res (fdb/inc n)
         entry [{:function "inc" :arguments n :result res} 10]
         _ (add-stack ?ctx entry)] res)))

(defn dec
  {:doc     "Decrements any number (or nil/null) by 1."
   :fdb/spec nil
   :fdb/cost 10}
  ([?ctx] (dec ?ctx 0))
  ([?ctx n]
   (let [res (fdb/dec n)
         entry [{:function "dec" :arguments n :result res} 10]
         _ (add-stack ?ctx entry)] res)))

(defn now
  {:doc      "Returns current epoch milliseconds on the executing machine."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (let [res     (.toEpochMilli (:instant ?ctx))
        entry   [{:function "now" :arguments [] :result res}  10]
        _       (add-stack ?ctx entry)] res))

(defn +
  {:doc      "Returns sum of each argument."
   :fdb/spec nil
   :fdb/cost "9 + count of numbers in +"}
  [?ctx & args]
  (let [res     (apply fdb/+ args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "+" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn -
  {:doc      "Returns difference of all the numbers in the sequence with the first number as the minuend."
   :fdb/cost "9 + count of numbers in -"
   :fdb/spec nil}
  [?ctx & args]
  (let [res     (apply fdb/- args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "-" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn *
  {:doc      "Returns product of all the numbers in the sequence."
   :fdb/spec nil
   :fdb/cost "9 + count of numbers in *"}
  [?ctx & args]
  (let [res     (apply fdb/* args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "*" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn /
  {:doc      "If no denominators are supplied, returns 1/numerator, else returns numerator divided by all of the denominators. Takes a sequence"
   :fdb/spec nil
   :fdb/cost "9 + count of numbers in /"}
  [?ctx & args]
  (let [res     (apply fdb// args)
        cost    (clojure.core/+ 9 (clojure.core/count [args]))
        entry   [{:function "/" :arguments [args] :result res} cost]
        _       (add-stack ?ctx entry)] res))

(defn quot
  {:doc      "Quot[ient] of dividing numerator by denominator."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx n d]
  (let [res     (fdb/quot n d)
        entry   [{:function "quot" :arguments [n d] :result res} 2]
        _       (add-stack ?ctx entry)] res))

(defn mod
  {:doc      "Modulus of num and div. Truncates toward negative infinity."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx n d]
  (let [res     (fdb/mod n d)
        entry   [{:function "mod" :arguments [n d] :result res} 2]
        _       (add-stack ?ctx entry)] res))

(defn rem
  {:doc      "Remainder of dividing numerator by denominator."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx n d]
  (let [res     (fdb/rem n d)
        entry   [{:function "rem" :arguments [n d] :result res} 2]
        _       (add-stack ?ctx entry)] res))

(defn boolean
  {:doc      "Coerce to boolean. Everything except false and nil is true."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx x]
  (let [res     (fdb/boolean x)
        entry   [{:function "boolean" :arguments x :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn re-find
  {:doc      "Coerce to boolean. Everything except false and nil is true."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx pattern string]
  (let [res     (fdb/re-find pattern string)
        entry   [{:function "re-find" :arguments [pattern string] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn valid-email?
  {:doc      "Determines whether an email is valid, based on its pattern"
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx email]
  (let [res     (fdb/valid-email? email)
        entry   [{:function "re-find" :arguments email :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn ?user_id
  {:doc      "Gets current user _id."
   :fdb/spec nil
   :fdb/cost "10 if no lookup necessary, 10 plus fuel cost if lookup necessary"}
  [?ctx]
  (let [[res fuel]   (cond (:user_id ?ctx)
                           [(:user_id ?ctx) 0]

                           (:auth_id ?ctx)
                           (fdb/?user_id-from-auth ?ctx)

                           :else
                           (raise ?ctx "Cannot access ?user_id from this function interface"))
        entry   [{:function "?user_id" :arguments "?ctx" :result res} (clojure.core/+ 10 fuel)]
        _       (add-stack ?ctx entry)] res))

(defn ?auth_id
  {:doc      "Gets current auth _id."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (if (:auth_id ?ctx)
    (let [res     (:auth_id ?ctx)
          entry   [{:function "?auth_id" :arguments "?ctx" :result res} 10]
          _       (add-stack ?ctx entry)] res)
    (raise ?ctx "Cannot access ?auth_id from this function interface")))

(defn objT
  {:doc      "Gets the summed object of all true flakes"
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (if (:flakes ?ctx)
    (let [res     (fdb/objT (:flakes ?ctx))
          entry   [{:function "objT" :arguments (:flakes ?ctx) :result res} 10]
          _       (add-stack ?ctx entry)] res)
    (raise ?ctx "Cannot access flakes, or use objT function from this function interface")))

(defn objF
  {:doc      "Gets the summed object of all false flakes"
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (if (:flakes ?ctx)
    (let [res     (fdb/objF (:flakes ?ctx))
          entry   [{:function "objF" :arguments (:flakes ?ctx) :result res} 10]
          _       (add-stack ?ctx entry)] res)
    (raise ?ctx "Cannot access flakes, or use objF function from this function interface")))

(defn flakes
  {:doc      "Gets the flakes from the current subject."
   :fdb/spec nil
   :fdb/cost 10}
  [?ctx]
  (if (:flakes ?ctx)
    (let [res     (:flakes ?ctx)
          entry   [{:function "flakes" :arguments "?ctx" :result res} 10]
          _       (add-stack ?ctx entry)] res)
    (raise ?ctx "Cannot access flakes from this function interface")))

(defn rand
  {:doc       "Using instant as a seed, returns a random number"
   :fdb/spec  nil
   :fdb/cost  10}
  [?ctx & args]
  (let [max'   (or ?ctx (first args) 10)
        seed  (.toEpochMilli (:instant ?ctx))
        res   (fdb/rand seed max')
        entry   [{:function "rand" :arguments [seed max] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn ceil
  {:doc       "Takes the ceiling of a number"
   :fdb/spec  nil
   :fdb/cost  10}
  [?ctx num]
  (let [res   (fdb/ceil num)
        entry   [{:function "" :arguments [ceil] :result res} 10]
        _       (add-stack ?ctx entry)] res))

(defn floor
  {:doc       "Takes the floor of a number"
   :fdb/spec  nil
   :fdb/cost  10}
  [?ctx num]
  (let [res   (fdb/floor num)
        entry   [{:function "" :arguments [floor] :result res} 10]
        _       (add-stack ?ctx entry)]
    res))
