(ns boot-hedge.aws.cloudformation-api
  (:require [boot-hedge.common.core :refer [now]]
            [camel-snake-kebab.core :refer [->camelCaseString ->PascalCase]])
  (:import [com.amazonaws.services.cloudformation AmazonCloudFormationClientBuilder]
           [com.amazonaws.regions Regions]
           [com.amazonaws.services.cloudformation.model DescribeStacksRequest]
           [com.amazonaws.services.cloudformation.model DescribeChangeSetRequest]
           [com.amazonaws.services.cloudformation.model AmazonCloudFormationException]
           [com.amazonaws.services.cloudformation.model CreateChangeSetRequest]
           [com.amazonaws.services.cloudformation.model ChangeSetType]
           [com.amazonaws.services.cloudformation.model ExecuteChangeSetRequest]
           [com.amazonaws.services.cloudformation.model Capability]
           [com.amazonaws.services.cloudformation.model Parameter]
           [com.amazonaws.waiters WaiterParameters]
           [com.amazonaws.util DateUtils]))

; this should be autogenerated
(def todo-changeset-name "TODO-change-this-asap")

; API endpoint communication
(defn client
  "Creates AmazonCloudFormationClient which is used by other API methods"
  []
  (-> (AmazonCloudFormationClientBuilder/standard)
   (.withRegion Regions/EU_WEST_3)
   (.build)))

(defn describe-stacks
  "Gets list of stacks or named stack"
  ([client]
   (.describeStacks client))
  ([client stack-name]
   (as-> (DescribeStacksRequest.) n
    (.withStackName n stack-name)
    (.describeStacks client n))))

(defn describe-changeset
  "Gets changeset info"
  [client changeset-id]
  (as-> (DescribeChangeSetRequest.) n
   (.withChangeSetName n changeset-id)
   (.describeChangeSet client n)))

(defn create-changeset
  "Creates new changeset with changeset request"
  [client changeset-request]
  (.createChangeSet client changeset-request))

(defn execute-changeset
  "Executes changeset"
  [client changeset-id]
  (as-> (ExecuteChangeSetRequest.) n
    (.withChangeSetName n changeset-id)
    (.executeChangeSet client n)))

(defn stack-exists?
  "Checks if given stack exists"
  [client stack-name]
  (try
    (let [result (describe-stacks client stack-name)
          size (count (.getStacks result))]
      (= 1 size))  
    (catch AmazonCloudFormationException e false)))

(defn stack-status
  "Gets status of given stack"
  [client stack-name]
  (try
    (let [result (describe-stacks client stack-name)
          stack (first (.getStacks result))]
      (.getStackStatus stack))
    (catch AmazonCloudFormationException e
      (throw (Exception. "API call failed" e)))))

(defn changeset-status
  "Gets status of given stack"
  [client changeset-id]
  (try
    (let [result (describe-changeset client changeset-id)
          status (.getStatus result)]
      status)
    (catch AmazonCloudFormationException e
      (throw (Exception. "API call failed" e)))))

(defn wait-for-changeset-create
  "Wait until changset is created"
  [client changeset-id]
  (-> client
      (.waiters)
      (.changeSetCreateComplete)
      (.run (-> (DescribeChangeSetRequest.)
                (.withChangeSetName changeset-id)
                (WaiterParameters.)))))

(defn wait-for-stack-create
  "Wait until stack is created"
  [client stack-name]
  (-> client
      (.waiters)
      (.stackCreateComplete)
      (.run (-> (DescribeStacksRequest.)
                (.withStackName stack-name)
                (WaiterParameters.)))))

(defn wait-for-stack-update
  "Wait until stack is updated"
  [client stack-name]
  (-> client
      (.waiters)
      (.stackUpdateComplete)
      (.run (-> (DescribeStacksRequest.)
                (.withStackName stack-name)
                (WaiterParameters.)))))

;logic
(defn create-or-update-stack
  "Construct stack create or update request and starts creation process"
  [client stack-name template type parameters]
  (let [description (str "Created with Hedge at " (DateUtils/formatISO8601Date (now)))
        create-or-update-stack-request 
        (-> (CreateChangeSetRequest.)
          (.withStackName stack-name)
          (.withChangeSetName todo-changeset-name) ; TODO make more unique
          (.withCapabilities ["CAPABILITY_IAM"])     ; TODO use types from SDK
          (.withChangeSetType type)
          (.withDescription description)
          (.withTemplateBody template)
          (.withParameters parameters))]
    (create-changeset client create-or-update-stack-request)))

(defn create-stack
  "Create new stack"
  [client stack-name template parameters]
  (create-or-update-stack client stack-name template ChangeSetType/CREATE parameters))

(defn update-stack
  "Update stack"
  [client stack-name template parameters]
  (create-or-update-stack client stack-name template ChangeSetType/UPDATE parameters))

; TODO: simplify this?
(defn parameters
  [& params]
  (let [parameters (apply hash-map params)
        temp (map (fn [[key val]] (-> (Parameter.)
                                      (.withParameterKey key)
                                      (.withParameterValue val)))
                  parameters)]
    temp))

(defn deploy-stack
  "Create or deploy stack with given name and template"
  [client stack-name template bucket key]
  (let [template-string (slurp template)
        stack-exists (stack-exists? client stack-name)
        parameters-for-deploy (parameters "FunctionDeploymentBucket" bucket 
                                          "FunctionDeploymentKey" key
                                          "PrettyDeploymentName" stack-name
                                          "DeploymentName" (->PascalCase stack-name))
        create-changeset-result ((if stack-exists update-stack create-stack) 
                                 client stack-name template-string parameters-for-deploy)
        changeset-id (.getId create-changeset-result)]
    (wait-for-changeset-create client changeset-id)
    (execute-changeset client changeset-id)
    (if stack-exists
      (wait-for-stack-update client stack-name)
      (wait-for-stack-create client stack-name))
    (println "Stack is ready!")
    (-> (describe-stacks client stack-name)
        (.getStacks)
        (first)
        (.getOutputs)
        ((fn [outputs]
          (doseq [x outputs]
            (println (.getDescription x) ":")
            (println (.getOutputValue x))))))))
