(ns boot-azure.azure
  (require [boot.util :as util]
           [clojure.java.io :as io]
           [pantomime.mime :refer [mime-type-of]]
           [camel-snake-kebab.core :refer :all]
           [camel-snake-kebab.extras :refer [transform-keys]])
  (import [com.microsoft.azure.storage CloudStorageAccount]
          [com.microsoft.azure.storage.blob BlobContainerPermissions BlobContainerPublicAccessType]))

(def account (System/getenv "AZURE_ACCOUNT"))
(def account-key (System/getenv "AZURE_ACCOUNT_KEY"))

(def storage-connection-conf
  {:default-endpoints-protocol :http
   :account-name account
   :account-key account-key})


(defn ->storage-connection-string [conf]
  (apply str
         (interleave
          (map
           (fn [[k v]] (str k "=" (name v)))
           (transform-keys ->PascalCaseString conf))
          (repeat ";"))))

(defn container [c]
  (->
   storage-connection-conf
   ->storage-connection-string
   CloudStorageAccount/parse
   (.createCloudBlobClient)
   (.getContainerReference c)))


(defn public-container-permission []
  (doto (BlobContainerPermissions.) (.setPublicAccess BlobContainerPublicAccessType/CONTAINER)))

(defn public-container [c]
  (doto
    (container c)
    (.createIfNotExists)
    (.uploadPermissions (public-container-permission))))


(defn block-blob-reference [c n]
  (.getBlockBlobReference c n))

(defn set-mime-type! [br f]
  (.setContentType (.getProperties br) (mime-type-of f))
  br)

(defn upload [br f]
  (let [fi (io/file f)]
    (.upload br (io/input-stream fi) (.length fi))))

(defn upload-file [c f n]
  (->
    (public-container c)
    (block-blob-reference n)
    (set-mime-type! f)
    (upload f)))
