(ns bhatkhande.parts
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.test.alpha :as stest]
            [bhatkhande.spec :as us]
            [quil.core :as q :include-macros true]
            [quil.middleware :as m]
            [quil.core :as q]))

(def swaramap
  (zipmap [:s :-r :r :-g :g :m :m+ :p :-d :d :-n :n :- :c]
          ["सा" "रे॒" "रे" "ग॒" "ग" "म" "म॑" "प" "ध॒" "ध" "नि॒" "नि" "-" "ऽ"]))

(defn disp-octave
  [dispinfo inp]
  (if (or
       (= :taar inp)
       (= :mandra inp))
    (let [disptext "·"
          {:keys [x y font-size text-align spacing octave] :as di} dispinfo
          y1 (if (= :taar inp) (- y octave) (+ y octave))]
      (q/text disptext x y1)
      dispinfo)
    dispinfo))

(defn disp-kan
  [dispinfo inp]
  (let [[oct note] inp
        ;_ (println " disp-kan " note " - " inp )
        {:keys [x y font-size spacing text-align kan-raise octave] :as di} dispinfo
        y1 (- y kan-raise)
        _ (q/text-size (- font-size 5))
        _ (q/text-align text-align)
        _ (disp-octave (assoc dispinfo :y y1 :octave (- octave 5)) oct)
        disptext (swaramap note)
        dip (update-in dispinfo [:x] (fn[i] (+ i (/ spacing 3) (q/text-width disptext))))
        ]
    (q/text disptext x y1)
    dip))

(defn disp-sam-khaali
  [dispinfo inp]
  (let [{:keys [x y sam-khaali font-size]} dispinfo
        {:keys [beat]} inp]
    (when beat
      (do
        (q/text-size (- font-size 5))
        (q/text beat x (+ y sam-khaali))))
    dispinfo))

(defn disp-bhaag
  [dispinfo inp]
  (let [{:keys [x y x-start x-end sam-khaali font-size spacing y-inc]} dispinfo
        {:keys [bhaag]} inp ]
    (if bhaag
      (let [ix x 
            [ix y1] (if (>= ix (* 0.9 x-end)) [x-start (+ y y-inc)]
                        [ix y])]
        (do
          (q/stroke-weight 2)
          (q/line [ix (- y1 sam-khaali)] [ix (+ y1 sam-khaali)])
          (assoc dispinfo :x (+ ix spacing (q/text-width bhaag)) :y y1)))
      dispinfo)))

(defn disp-meend
  [dispinfo ]
  (let [over-text "︵"
        {:keys [x y over]} dispinfo]
    (do
      (q/text over-text x (- y over)))
    dispinfo))

(defn disp-swara
  [dispinfo inp]
  (let [{:keys [::us/note ::us/kan ::us/khatka ::us/meend-start ::us/meend-end bhaag]} inp
        dispinfo (if kan (disp-kan dispinfo kan) dispinfo)
        dispinfo (if meend-start (disp-meend dispinfo) dispinfo)
        {:keys [x y font-size spacing text-align] :as di} dispinfo
        _ (disp-sam-khaali dispinfo inp)
        _ (q/text-size font-size)
        _ (q/text-align text-align)
        disptext (swaramap (note 1))
        _ (disp-octave dispinfo (note 0))
        dip (update-in dispinfo [:x] (fn[i] (+ i spacing (q/text-width disptext))))]
    (q/text disptext x y)
    dip))

(defn disp-note
  [dispinfo inp]
  (let [{:keys [::us/note bhaag]} inp]
    (if bhaag
      (disp-bhaag dispinfo inp)
      (disp-swara dispinfo inp))))

(defn disp-underbrace
  [dispinfo ]
  (let [ under-m "︶"
        {:keys [x y under]} dispinfo]
    (q/text under-m x (+ y under))))


(defn disp-s-note
  "show an s-note"
  [dispinfo inp]
  (if (= 1 (count inp))
    (disp-note dispinfo (first inp))
    (let [ispa (:spacing dispinfo)
          _ (disp-underbrace dispinfo)
          res (reduce disp-note (assoc dispinfo :spacing 0) inp)]
      ;;remove spacing so that the swaras as close together
      (assoc res :spacing ispa :x (+ ispa (:x res))))))

(defn disp-m-note
  [dispinfo inp]
  (reduce disp-s-note dispinfo inp))

(defn disp-part-label
  [dispinfo inp]
  (let [{:keys [part-header-font-size x y header-y-spacing]} dispinfo]
    (q/text-size part-header-font-size)
    (q/text inp x y)
    (assoc dispinfo :y (+ y header-y-spacing))))

(defn split-into-bhaags
  [inp bhaags]
  (let [redfn (fn[{:keys [pre post] :as m} i]
                (let [[f1 f2] (split-at i post)]
                  {:pre (conj pre (vec f1)) :post f2}))
        res (reduce redfn {:pre [] :post inp}
                    bhaags)
        intr (repeat (count bhaags) [[{:bhaag "|"}]])]
    (reduce into (->> (->> (:pre res) (remove empty?) vec) 
                      (interleave intr)))))

(defn append-bhaags
  [{:keys [::us/m-noteseq ::us/taal ::us/part-label] :as m}]
  (let [{:keys [::us/num-beats ::us/sam-khaali ::us/bhaags]} taal]
    (update-in m [::us/m-noteseq]
               #(->> % 
                     (partition-all num-beats )
                     (mapv (fn[i] (split-into-bhaags i bhaags)))
                     (reduce into)))))

(defn add-sam-khali
  [{:keys [::us/m-noteseq ::us/taal ::us/part-label] :as m}]
  (let [sam-khaali (taal ::us/sam-khaali)
        nb (taal ::us/num-beats)]
    (assoc m ::us/m-noteseq
           (mapv #(if-let [iv (sam-khaali %2)]
                    (update-in %1 [0] (fn[i](assoc i :beat iv)))
                    %1)
                 m-noteseq
                 (iterate #(if (> nb %) (inc %) 1) 1)))))

(defn line-separator
  [dispinfo]
  (let [{:keys [x y x-end header-y-spacing]} dispinfo
        x2 (/ x-end 2)
        x3 (/ x-end 4)]
    (q/line [(- x2 x3 ) (+ y header-y-spacing)]
            [(+ x3 x2) (+ y header-y-spacing)])))

(defn disp-part
  [dispinfo inp]
  (let [{:keys [x y x-start header-y-spacing]} dispinfo
        {:keys [::us/m-noteseq ::us/taal ::us/part-label]}
        (-> inp add-sam-khali append-bhaags)
        d1 
        (-> dispinfo
            (disp-part-label part-label)
            (disp-m-note m-noteseq))]
    (line-separator d1)
    (assoc d1 :x x-start :y (+ (:y d1) (* 2 header-y-spacing)))))

(defn disp-comp-label
  [dispinfo inp]
  (let [{:keys [x y  comp-label-font-size header-y-spacing]} dispinfo]
    (q/text-size comp-label-font-size)
    (q/text inp x y)
    (line-separator dispinfo)
    (assoc dispinfo :y (+ y (* 2 header-y-spacing)))))

(defn disp-comp
  [dispinfo inp]
  (let [{:keys [::us/parts ::us/taal ::us/comp-label ::us/comp-id]} inp
        d1 (-> dispinfo
               (disp-comp-label comp-label))
        inp2 (mapv #(if (% ::us/taal) %
                        (assoc % ::us/taal taal))
                   parts)]
    (reduce disp-part d1 inp2)))
