(ns sgr-sound.converter
  (:require [sgr-sound.utils :refer  :all]
            [sgr-mfcc.mfcc   :as     MFCC])
  (:import  [javax.sound.sampled AudioSystem AudioFormat AudioFileFormat AudioFileFormat$Type AudioFormat$Encoding]))

(def SAMPLE_RATE    16000.0)
(def SAMPLE_SIZE_IN_BITS 16)
(def CHANNELS             1)
(def SIGNED            true)
(def BIG_ENDIAN       false)
(def AUDIO_FILE_FORMAT_TYPE AudioFileFormat$Type/WAVE)

(defn convert
  {:doc
"default:
   SAMPLE_RATE    16000.0
   SAMPLE_SIZE_IN_BITS 16
   CHANNELS             1
   SIGNED            true
   BIG_ENDIAN       false
   AUDIO_FILE_FORMAT_TYPE AudioFileFormat$Type/WAVE

return:
   success -> targetFile
   error   -> false"}
  [sourceFile targetFile
   & {:keys [sampleRate sampleSizeInBits channels signed bigEndian audioFileFormatType]
      :or   {sampleRate          SAMPLE_RATE
             sampleSizeInBits    SAMPLE_SIZE_IN_BITS
             channels            CHANNELS
             signed              SIGNED
             bigEndian           BIG_ENDIAN
             audioFileFormatType AUDIO_FILE_FORMAT_TYPE}}]
  (try
    (let [sourceFormat (getFormat (getAudioFileFormat sourceFile))
          targetFormat (newAudioFormat sampleRate sampleSizeInBits channels signed bigEndian)]
      (if (isConversionSupported targetFormat sourceFormat)
        (let [sourceStream (getAudioInputStream sourceFile)
              targetStream (getAudioInputStream targetFormat sourceStream)]
          (writeAudioStream targetStream audioFileFormatType targetFile)
          targetFile)
        false))
    (catch Exception e
      (do
        (println (str "Exception SGR-sound.converter.convert: " (.getMessage e)))
        false))))

(def NUMBERS_OF_COEFF   17)
(def WINDOW_SIZE       512)
(def USE_FIRST_COEFF false)
(def MIN_FREQ         20.0)
(def MAX_FREQ      16000.0)
(def NUMBER_OF_FILTERS 250)


(defn sound->convert->mfcc
  {:doc
"default:
   SAMPLE_RATE    16000.0
   SAMPLE_SIZE_IN_BITS 16
   CHANNELS             1
   SIGNED            true
   BIG_ENDIAN       false
   AUDIO_FILE_FORMAT_TYPE AudioFileFormat$Type/WAVE
   NUMBERS_OF_COEFF    27
   WINDOW_SIZE        512
   USE_FIRST_COEFF  false
   MIN_FREQ          20.0
   MAX_FREQ       16000.0
   NUMBER_OF_FILTERS  250

return:
   {:sourceFile   sourceFile
    :targetFile   targetFile
    :convertFile  success -> convertFile || error -> false
    :mfcc         success -> mfcc        || error -> false
   }"}
  [sourceFile targetFile
   & {:keys [sampleRate sampleSizeInBits channels signed bigEndian audioFileFormatType
             windowSize numberOfCoeff useFirstCoeff minFreq maxFreq numberOfFilters]
      :or {sampleRate          SAMPLE_RATE
           sampleSizeInBits    SAMPLE_SIZE_IN_BITS
           channels            CHANNELS
           signed              SIGNED
           bigEndian           BIG_ENDIAN
           audioFileFormatType AUDIO_FILE_FORMAT_TYPE
           windowSize          WINDOW_SIZE
           numberOfCoeff       NUMBERS_OF_COEFF
           useFirstCoeff       USE_FIRST_COEFF
           minFreq             MIN_FREQ
           maxFreq             MAX_FREQ
           numberOfFilters     NUMBER_OF_FILTERS}}]
  (try
    (let [convertFile (convert sourceFile targetFile
                               :sampleRate       sampleRate
                               :sampleSizeInBits sampleSizeInBits
                               :channels         channels
                               :signed           signed
                               :bigEndian        bigEndian
                               :audioFileFormatType audioFileFormatType)]
      (if (false? convertFile)
        {:sourceFile  sourceFile
         :targetFile  targetFile
         :convertFile false
         :mfcc        false}
        (let [mfcc (MFCC/mfcc convertFile windowSize numberOfCoeff useFirstCoeff minFreq maxFreq numberOfFilters)]
          {:sourceFile  sourceFile
           :targetFile  targetFile
           :convertFile convertFile
           :mfcc        mfcc})))
    (catch Exception e
      (do
        (println (str "Exception SGR-sound.converter.sound->convert->mfcc: " (.getMessage e)))
         {:sourceFile  sourceFile
          :targetFile  targetFile
          :convertFile false
          :mfcc        false}))))


(defn sound->convert->mfcc->average
  {:doc
"default:
   SAMPLE_RATE    16000.0
   SAMPLE_SIZE_IN_BITS 16
   CHANNELS             1
   SIGNED            true
   BIG_ENDIAN       false
   AUDIO_FILE_FORMAT_TYPE AudioFileFormat$Type/WAVE
   NUMBERS_OF_COEFF    27
   WINDOW_SIZE        512
   USE_FIRST_COEFF  false
   MIN_FREQ          20.0
   MAX_FREQ       16000.0
   NUMBER_OF_FILTERS  250

return:
   {:sourceFile   sourceFile
    :targetFile   targetFile
    :convertFile  success -> convertFile || error -> false
    :mfcc         success -> mfcc        || erro  -> false
   }"}
  [sourceFile targetFile
   & {:keys [sampleRate sampleSizeInBits channels signed bigEndian audioFileFormatType
             windowSize numberOfCoeff useFirstCoeff minFreq maxFreq numberOfFilters]
      :or {sampleRate          SAMPLE_RATE
           sampleSizeInBits    SAMPLE_SIZE_IN_BITS
           channels            CHANNELS
           signed              SIGNED
           bigEndian           BIG_ENDIAN
           audioFileFormatType AUDIO_FILE_FORMAT_TYPE
           windowSize          WINDOW_SIZE
           numberOfCoeff       NUMBERS_OF_COEFF
           useFirstCoeff       USE_FIRST_COEFF
           minFreq             MIN_FREQ
           maxFreq             MAX_FREQ
           numberOfFilters     NUMBER_OF_FILTERS}}]
  (try
    (let [result (sound->convert->mfcc sourceFile targetFile
                                       :sampleRate       sampleRate
                                       :sampleSizeInBits sampleSizeInBits
                                       :channels         channels
                                       :signed           signed
                                       :bigEndian        bigEndian
                                       :audioFileFormatType audioFileFormatType
                                       :windowSize      windowSize
                                       :numberOfCoeff   numberOfCoeff
                                       :useFirstCoeff   useFirstCoeff
                                       :minFreq         minFreq
                                       :maxFreq         maxFreq
                                       :numberOfFilters numberOfFilters)
          convertFile (:convertFile result)
          mfcc        (:mfcc result)]
      (if (or (false? convertFile) (false? mfcc))
        result
        (let [averagemfcc (MFCC/average-mfcc mfcc)]
          {:sourceFile  sourceFile
           :targetFile  targetFile
           :convertFile convertFile
           :mfcc        averagemfcc})))
    (catch Exception e
      (do
        (println (str "Exception SGR-sound.converter.sound->convert->mfcc->average: " (.getMessage e)))
         {:sourceFile  sourceFile
          :targetFile  targetFile
          :convertFile false
          :mfcc        false}))))
