(ns sgr-db.user
  (:require [monger.core       :as        mg]
            [monger.collection :as        mc]
            [monger.util       :as      util]
            [monger.result     :refer  [ok?]]
            [monger.joda-time  :as joda-time]
            [monger.util         :refer :all]
            [clj-time.core     :as      time]
            [schema.core       :as         s]

            [sgr-db.config     :refer :all]
            [sgr-db.models     :as    models]))

(defn- insert-user
  "Place a new user in the database. Must be a valid user, returns true if successfully inserted, otherwise false."
  [dbname coll user]
  (let [db (mg/get-db conn dbname)]
    (try
      ;;(mc/ensure-index db coll (array-map :index 1) { :unique true })
      (ok? (mc/insert db coll (s/validate models/User user)))
      (catch Exception e
        (do
          (println (str "Exception SGR-db.user.insert-user: " (.getMessage e)))
          false)))))

(defn- update-user
  "example:
  (update-user 'mongo-test' 'users' {:user-id 'd2708627-1f44-4b12-9bd6-4df906cc6a37'} {'$set' {:gender 'male'}})"
  [dbname coll conditions document]
  (let [db (mg/get-db conn dbname)]
    (ok? (mc/update db coll conditions document))))

(defn get-users
  ([dbname coll]
   (let [db (mg/get-db conn dbname)]
     (if-let [users (mc/find db coll)]
       users)))
  ([dbname coll query]
   (let [db (mg/get-db conn dbname)]
     (if-let [users (mc/find db coll query)]
       users)))
  ([dbname coll query fields]
   (let [db (mg/get-db conn dbname)]
     (if-let [users (mc/find db coll query fields)]
       users))))

(defn get-users-maps
  ([dbname coll]
   (let [db (mg/get-db conn dbname)]
     (if-let [users (mc/find-maps db coll)]
       users)))
  ([dbname coll query]
   (let [db (mg/get-db conn dbname)]
     (if-let [users (mc/find-maps db coll query)]
       users)))
  ([dbname coll query fields]
   (let [db (mg/get-db conn dbname)]
     (if-let [users (mc/find-maps db coll query fields)]
       users))))

(defn get-user
  [dbname coll query]
  (let [db (mg/get-db conn dbname)]
    (if-let [user (mc/find-one db
                               coll
                               query)]
      user)))


(defn get-user-as-map
  [dbname coll query]
  (let [db (mg/get-db conn dbname)]
    (if-let [user (mc/find-one-as-map db
                                      coll
                                      query)]
      user)))

(defn get-user-gender
  {:doc "<conditions or user>"}
  [dbname coll conditions]
  (:gender (get-user-as-map dbname coll conditions)))

(defn set-user-gender
  {:doc "<conditions or user>"}
  [dbname coll conditions gender]
  (update-user dbname coll conditions {"$set" {:gender gender}}))

(defn get-user-genderSVM
  {:doc "<conditions or user>"}
  [dbname coll conditions]
  (:genderSVM (get-user-as-map dbname coll conditions)))

(defn set-user-genderSVM
  {:doc "<conditions or user>"}
  [dbname coll conditions genderSVM]
  (update-user dbname coll conditions {"$set" {:genderSVM genderSVM}}))

(defn get-user-name
  {:doc "<conditions or user>"}
  [dbname coll conditions]
  (:name (get-user-as-map dbname coll conditions)))

(defn set-user-name
  {:doc "<conditions or user>"}
  [dbname coll conditions name]
  (update-user dbname coll conditions {"$set" {:name name}}))

(defn get-user-SVM?
  {:doc "<conditions or user>"}
  [dbname coll conditions]
  (:SVM? (get-user-as-map dbname coll conditions)))

(defn set-user-SVM?
  {:doc "<conditions or user>"}
  [dbname coll conditions SVM]
  (update-user dbname coll conditions {"$set" {:SVM? SVM}}))

(defn push-user-record
  {:doc "<conditions or user>"}
  [dbname coll conditions record]
  (update-user dbname coll conditions {"$push" {:records record}}))

(defn push-user-records
  {:doc "<conditions or user>"}
  [dbname coll conditions records]
  (update-user dbname coll conditions  {"$pushAll" {:records records}}))


(defn create-user
  {:doc
   ""}
  [dbname coll & {:keys [name gender genderSVM SVM?]
                   :or  {name "unknown" gender "unknown" genderSVM "unknown" SVM? false}}]
   (try
     (let [user {:user-id    (random-uuid)
                 :records    []
                 :created-on (time/now)
                 :name       name
                 :gender     gender
                 :genderSVM  genderSVM
                 :SVM?       SVM?}]
       (if (insert-user dbname coll user)
         user
         false))
     (catch Exception e
       (do
         (println (str "Exception SGR-db.user.create-user: " (.getMessage e)))
         false))))


