(ns surfer.ckan
  (:require
    [clj-http.client :as client]
    [surfer.store :as store]
    [surfer.utils :as utils]
    [clojure.data.json :as json]
    [clojure.data.csv :as csv]
    [clojure.string :as str]
    [clojure.java.io :as io]
    )
  (:import [java.time Instant]
           [java.util Date]))

(def ^:dynamic *import-userid* nil)

(def PROVENANCE_PREFIXES
  {
   "xsd" "http://www.w3.org/2001/XMLSchema#",
   "prov" "http://www.w3.org/ns/prov#",
   "ocn" "http://www.oceanprotocol.com/prov"
  })

(defn create-provenance 
  "Creates import provenance from a CKAN asset."
  [pack options]
  (let [importer-label "ocn:importer"
        activity-label "ocn:import"
        entity-label "ocn:this"] 
    {:prefix PROVENANCE_PREFIXES
     :agent {importer-label {
                             "prov:label" "Ocean CKAN Importer"
                             "prov:type" "ocn:import"
                             }}
     :activity {"ocn:import" {"prov:label" "Compile the dataset",
                               "prov:endTime" (Instant/now)
                               "prov:type" "ocn:import"
                               }}
     :entity  {entity-label  {"prov:value"  {"$" (or (:name pack) "Unnamed dataset"),
                                             "type" "xsd:string"}
                              "prov:type" "ocn:dataset",
                              ;; "ocn:id_resolver"   "did:ocn:marketplaceid"
                              }}
     "wasGeneratedBy" {"_:wGB4" {"prov:entity" entity-label,
                                 "prov:activity" activity-label
                                 }}
     "wasAssociatedWith" {"_:wAW4" {"prov:agent" importer-label,
                                    "prov:activity" activity-label
                                    }}
     })) 

(defn convert-meta
  "Converts metadata from a CKAN asset to a Ocean Asset"
  ([pack options]
    (let [base {:name (:name pack)
                :description (:description pack) 
                :license (:license_title pack)
                :tags (seq (map :name (:tags pack)))
                :type "dataset" 
                :provenance (create-provenance pack options) 
                }
          meta base] 
      (utils/remove-nil-values meta))))

(defn api-call 
  "Makes a call to the CKAN API with a given URL"
  ([url]
  (let [body (:body (client/get url))
        json (json/read-str body :key-fn keyword)]
    (json :result))))

(defn tag-list 
  "Gets a list of tags from a CKAN repo"
  ([repo]
    (let [url (str repo "/api/3/action/tag_list")]
      (api-call url)))) 

(defn package-list 
  "Gets a list of packages from a CKAN repo"
  ([repo]
    (let [url (str repo "/api/3/action/package_list")]
      (api-call url)))) 

(defn package-show 
  "Gets CKAN package metadata for a given package ID"
  ([repo package]
    (let [url (str repo "/api/3/action/package_show?id=" package)]
      (api-call url))))

(defn import-package 
  "Import a package from the CKAN repo and list on the marketplace"
  ([repo package-name]
    (import-package repo package-name {}))
  ([repo package-name options]
    (let [pdata (package-show repo package-name)
          odata (convert-meta pdata options) ;; base metadata conversion
          extra-data (utils/remove-nil-values ;; extra fields related to import
                       {:dateCreated (Instant/now)})
          odata (merge odata extra-data)
          assetid (store/register-asset (json/write-str odata))]
      (store/create-listing 
        {:assetid assetid
         :status "published"
         :info (utils/remove-nil-values {:title (or (:title pdata) (:name odata))
                                         :type (:type pdata)
                                         :description (:description odata)
                                         :price "0"
                                         :source (str "This asset was automatically generated by import from the CKAN repository " repo " at " (str (java.time.Instant/now)))})
         :userid (or *import-userid* (throw (IllegalArgumentException. "*import-userid* must be bound to import CKAN asset")))})))) 

(defn import-packages [repo package-names]
  (doall (pmap #(import-package repo % ) package-names)))

(defn import-all [repo]
  (let [packages (package-list repo)]
    (doseq [p packages] (import-package repo p))
    (println (str (count packages) " packages imported")))) 

(defn get-data 
  "Gets the data for a given resource"
  [resource]
  ((client/get (resource "url")) :body))

(defn ludo-meta [pack]
  (let [ne #(if (empty? %) nil %) ;; helper function to detect non-empty elements
        base {"name" (or (ne (:name pack)) 
                         "No name specified")
              "author" (or (ne (:author pack)) 
                           (ne (:name (:organization pack))) 
                           (ne (:description (:organization pack))) 
                           "No author specified")
              "dateCreated" (.toString (Instant/now))
              "license" (or (ne (:license_id pack)) 
                            "No License Specified")
              "price" 0
              "tags" (str/join "," (mapv :display_name (:tags pack)))
              "type" "dataset"
              "description" (or (ne (:description pack)) "No description")
              }
        files (map :url (:resources pack))
        meta (reduce (fn [m [i url]]
                        (assoc m (str "files[" i "]:url") url)) 
                     base
                     (take 5 (zipmap (next (range)) files)))]
    meta))

(defn ludo-export 
  ([repo]
    (ludo-export repo 10))
  ([repo num]
    (let [package-names (take num (package-list repo))
          packages (map #(package-show repo %) package-names)
          exports (map ludo-meta packages)
          headers ["name" 
                   "<local>" 
                   "<process>"
                   "<set public>"
                   "<register>"
                   "<reset>"
                   "<delete>"
                   "dateCreated"
                   "author"
                   "license"
                   "contentType"
                   "price"
                   "categories"
                   "tags"
                   "type"
                   "description"
                   "copyRightHolder"
                   "encoding"
                   "compression"
                   "workExample"
                   "inLanguage"
                   "files[1]:url"
                   "files[2]:url"
                   "files[3]:url"
                   "files[4]:url"
                   "files[5]:url"]
          rows (mapv (fn [ex]
                       (for [h headers] (or (ex h) ""))) exports)
          output (apply mapv vector (cons headers rows))]
      (with-open [writer (io/writer "output.csv")]
      (csv/write-csv writer output))))
  )