(ns grave.handlers
  (:use rip.core
        grave.util
        rip.util
        rip.validation
        rip.middleware
        hiccup.page
        grave.validation)
  (:require [com.twinql.clojure.conneg :as conneg]
            [noir.response :as resp]))

(defmacro edit
  [scope & body]
  `(GET ~scope {:name :edit :path "/:id/edit"} ~@body))

(defmacro new*
  [scope & body]
  `(GET ~scope {:name :new :path "/new"} ~@body))

;; Midleware

(defn with-form-validator
  "Wrap a scope with a validation to be used with if-valid-form."
  [scope validator param-key]
  (fn [handler]
    (fn [request]
      (let [value (get-in request [:params param-key])]
        (if value
          (let [value      (transform-assocs validator value)
                validation (-> (validate validator value)
                               (assoc
                                   :form-value
                                 (transform-assocs validator value true))
                               (update-in [:errors] form-errors validator))]
            (binding [*validation* validation]
              (handler request)))
          (throw (Exception. "Parameter can't be nil")))))))

(defn with-layout
  [layout]
  (fn [h]
    (fn [r]
      (let [response (h r)]
        (if (= (type response) :view)
          (layout response)
          response)))))

;; Responses

(defmacro dispatch
  [& clauses]
  `(let [ctypes# (set
                  (conneg/best-allowed-content-type
                   (get-in *request* [:headers "accept"] "text/plain")
                   #{"text/html" "application/json" "application/xml"}))]
     (condp (fn [ctype# ctypes#]
              (contains? ctypes# (name ctype#))) ctypes#
       ~@clauses)))

(defn created
  [url & [response]]
  (-> (or response {:body ""})
      (assoc-in [:headers "Location"] url)
      (assoc :status 201)))

(defn flash
  [value & [response]]
  (-> (or response {:status 201 :body ""})
      (assoc :flash value)))

(defn unprocessable-entity
  [& [response]]
  (-> (or response {:body ""})
      (assoc :status 422)))
