;; copyright (c) 2016 Sean Corfield

(ns engine.committable
  "Committable protocol for output data sources.
  Also commit! implementation for update! or insert!
  selection, based on primary key in the row."
  (:require [engine.input :as i]))

(defprotocol Committable
  "An output data source is committable. You can delete! (by key
  or key set), insert! new data (rows), and update! existing data
  (rows by key), it also knows how to get / lookup / generate the
  primary key for any given table."
  (delete! [this table pk v])
  (insert! [this table row])
  (update! [this table row pk v])
  (key-generator [this table])
  (primary-key [this table])
  (lookup-key [this table]))

(defrecord Insertable [context inserter]
  ;; default implementations either throw an exception...
  ;; ...or return nil, to make it less onerous to write
  ;; new Committable resources as instances of Insertable
  Committable
  (delete! [this table pk v]
    (throw (ex-info "delete! not supported on this data source."
                    {:type (type context) :inserter (type inserter)
                     :table table :pk pk :v v})))

  (insert! [this table row]
    (inserter context table row))

  (update! [this table row pk v]
    (throw (ex-info "update! not supported on this data source."
                    {:type (type context) :inserter (type inserter)
                     :table table :row-keys (keys row) :pk pk :v v})))

  (key-generator [this table] nil)
  (primary-key [this table] nil)
  (lookup-key [this table] nil))

(defn insertable-store*
  "Given an insertion function, or a context (probably a Component)
  and an insertion function, return an Insertable data store."
  ([inserter] (insertable-store* nil inserter))
  ([context inserter] (->Insertable context inserter)))

(defmacro insertable-store
  "Usage:
  (insertable-store [this table row] body)
  or:
  (insertable-store context [this table row] body)
  Returns an Insertable."
  [params & body]
  (if (vector? params)
    `(insertable-store* (fn ~params ~@body))
    (let [context params
          [params & body] body]
      `(insertable-store* ~context (fn ~params ~@body)))))

(defn lookup-keys
  "For any row fields that are keywords, look up their value
  in the environment. This allows updates to refer to keys
  generated by earlier updates in the list."
  [row env]
  (reduce-kv (fn [row k v] (assoc row k (if (keyword? v) (v env) v)))
             {} row))

(defn commit!
  "Commit all of the given updates to the supplied data sources.
  Applies the updates in order, building the key lookup environment
  as it goes, and replacing key references (columns with keyword values).
  Although we generally assume tables, and rows, the table is really
  just an arbitrary key into the data store and the row could be any
  type of value. If the row is not a map, we only ever do insert!
  If primary-key returns a non-nil value, we assume key generation
  is in effect and will attempt to lookup foreign keys in the
  environment, based on any previously seen keys for updates."
  [data-sources updates]
  (reduce (fn [env [key dsn table row pk key-gen delete-key]]
            (let [ds (i/get-dsn data-sources dsn)
                  pk (or pk (primary-key ds table))
                  key-gen (or key-gen (key-generator ds table) identity)
                  key-find (lookup-key ds table)]
              (if delete-key
                (do
                  (delete! ds table pk delete-key)
                  env)
                (if-let [pkv (and pk (map? row) (get row pk))]
                  (do
                    (update! ds table (lookup-keys (dissoc row pk) env) pk pkv)
                    (cond-> env key (assoc key pkv)))
                  (if-let [found-pk (and pk key-find (key-find ds row))]
                    (do
                      (update! ds table (lookup-keys row env) pk found-pk)
                      (cond-> env key (assoc key found-pk)))
                    (let [new-row (key-gen row)
                          new-pk (insert! ds table (cond-> new-row
                                                     pk (lookup-keys env)))]
                      (cond-> env key (assoc key new-pk))))))))
          {}
          updates))
