(ns schopfhirsch.hash-router)

(defonce ^:private router (atom nil))

(defn regex-lookup [routes hash]
  (loop [route routes]
    (if (nil? route)
      {:route {:enter (fn [] (js/console.error "Schopfhirsch Hash Router: No matching route found:" hash))
               :args []}}
      (if-let [matches (re-matches (:rule (first route)) hash)]
        {:route (first route)
         :matches matches}
        (recur (next route))))))

(defn function-calling-resolve [arg matches]
  (if (number? arg)
    (get matches arg)
    (arg)))

(defonce ^:private resolver
  ;; A resolver that handles numbers as placeholders for regex
  ;; matches. Other arguments will be called as functions and their
  ;; return value is used as argument value.
  (atom {:lookup regex-lookup
         :resolve function-calling-resolve}))

(defn ^:private current-hash []
  ;; Returns the hash part of the current URL.
  (-> js/document .-location .-hash))

(defn ^:private provide-args [route matches]
  ;; Applies the resolver function for the given route.
  (map (fn [arg]
         ((:resolve @resolver) arg matches))
       (:args route)))

(defn ^:private hash-change-handler [state routes]
  (fn [& rest]
    (swap! state
           (fn [old-state]
             (let [new-route ((:lookup @resolver) @routes (current-hash))
                   params (provide-args (:route new-route) (:matches new-route))]
               (when-let [exit-fun (:exit (:route (:route old-state)))]
                 (apply exit-fun (:params old-state)))
               (when-let [enter-fun (:enter (:route new-route))]
                 (apply enter-fun params))
               (assoc old-state
                      :params params
                      :route new-route))))))

(defn ^:private flatten-routes [routes]
  (vec (flatten routes)))

(defn ^:private new-router [initial-routes]
  (let [routes (atom (flatten-routes initial-routes))
        current-hash (current-hash)
        initial-route ((:lookup @resolver) @routes current-hash)
        {route :route matches :matches} initial-route
        state (atom {})
        change-handler (hash-change-handler state routes)]
    {:start (fn []
              (change-handler)
              (js/window.addEventListener "hashchange" change-handler))
     :change-routes (fn [new-routes]
                      (swap! routes #(flatten-routes new-routes)))}))

(defn ^:export set-routes! [routes]
  (swap! router (fn [old-router]
                  (if (nil? old-router)
                    (let [router (new-router routes)]
                      ((:start router))
                      router)
                    (let [change-fun (:change-routes old-router)]
                      (change-fun routes)
                      old-router)))))

(defn ^:export swap-resolver! [new-resolver]
  (swap! resolver new-resolver))
