(ns google-apis.v4.sheets
  (:require [clj-http.client :as http]
            [cheshire.core :as json]
            [clojure.string :as str]))


(def ^:no-doc errors
  {:sheet-tab-name "Error at google-apis.v4.sheets/get-sheet\nPlease add 'Sheet tab name' parameter :sheet-tab-name,\nwe couldn't send the request without this!"
   :sheets-id      "Error at google-apis.v4.sheets/get-sheet\nPlease add Google Spreadsheet's 'Sheets ID' parameter :sheets-id,\nwe couldn't send the request without this!"})

(defn ^:no-doc if-missing-stop!
  ([sheets-id]
   (if-missing-stop! sheets-id true))
  ([sheets-id sheet-tab-name]
   (if (nil? sheet-tab-name) (assert false (:sheet-tab-name errors)))
   (if (nil? sheets-id) (assert false (:sheets-id errors)))))


(defn get-tab-names
  "Get spreadsheet tab names related to one sheets ID"
  {:added "0.1.0"}
  [{{:keys [api-key bearer]} :creds
    {:keys [sheets-id]}      :query}]
  (if-missing-stop! sheets-id)
  (:sheets
    (json/parse-string
      (:body
        (http/get (str "https://sheets.googleapis.com/v4/spreadsheets/" sheets-id "?key=" api-key)
                  {:accept           :json
                   :throw-exceptions false
                   :headers          {"Authorization" (str "Bearer " bearer "")}
                   :compressed       true}))
      true)))


(defn get-sheet
  "**Read a single range**\n
   The following spreadsheets.values.get request reads the values stored in the range
   Sheet1!A1:D5 and returns them in the response. Empty trailing rows and columns are
   omitted.\n\nThe request protocol is shown below. The Reading and Writing Values guide
   shows how to implement reads in different languages using the Google API client libraries.\n
   Google API Reference doc [link](https://developers.google.com/sheets/api/samples/reading)\n\n

   Min requirements:
    api-key,
    bearer token,
    sheets ID,
    sheet tab name

   \n
    (get-sheet {:creds {:api-key api-key
                        :bearer  bt}
                :query {:sheets-id      \"1gQ1v********************sccBqtp4\"
                        :sheet-tab-name \"Your-heet-tab-name1\"
                        :range-of       [\"A1\" \"C1\"]
                        :from-response  [:body :headers :status]}})
    "
  {:added "0.1.0"
   :doc/format :markdown}
  [{{:keys [api-key bearer]}        :creds
    {:keys [sheets-id sheet-tab-name
            range-of from-response]
     :or   {range-of      false
            from-response [:body]}} :query}]
  (if-missing-stop! sheets-id sheet-tab-name)
  (let [range (if range-of
                sheet-tab-name
                (str sheet-tab-name "!" (str/join ":" range-of)))]
    (select-keys
      (http/get (str "https://sheets.googleapis.com/v4/spreadsheets/" sheets-id "/values/" range "?key=" api-key)
                {:accept           :json
                 :throw-exceptions false
                 :headers          {"Authorization" (str "Bearer " bearer "")}
                 :compressed       true})
      from-response)))

(defn post-sheet
  [] )



(comment

  ; old version

  (defn get-sheet
    ([bt api-key id sheet-name] (get-sheet bt api-key id sheet-name {:range-of nil}))
    ([bt api-key id sheet-name sheet]
     (let [range (if-not (:range-of sheet) sheet-name (str sheet-name "!" (:range-of sheet)))]
       (http/get (str "https://sheets.googleapis.com/v4/spreadsheets/" id "/values/" range "?key=" api-key)
                 {:accept           :json
                  :throw-exceptions false
                  :headers          {"Authorization" (str "Bearer " bt "")}
                  :compressed       true})))))

