(ns google-apis.auth.service-account
  (:require [cemerick.url :as url]
            [cheshire.core :as json]
            [clj-jwt.core :as jwt]
            [clj-jwt.key :as key]
            [clj-time.core :as time]
            [clj-http.client :as http]
            [clojure.string :as str]
            [clojure.edn :as edn]
            [gapi.core :as g]
            [gapi.auth]
            [environ.core :refer [env]])
  (:import java.io.StringReader))


;; idea from https://github.com/arohner

(def ^:no-doc token (atom ""))                                       ; oauth2 token
(def ^:no-doc v3-token (atom ""))                                    ; oauth2 token

(defn- load-creds
      "Takes a path to a service account .json credentials file"
      [secrets-json-path]
      (-> secrets-json-path slurp (json/parse-string keyword)))


(defn- create-claim [scopes creds & [{:keys [sub] :as opts}]]
      (let [claim (merge {:iss   (:client_email creds)
                          :scope (str/join " " scopes)
                          :aud   "https://www.googleapis.com/oauth2/v4/token"
                          :exp   (-> 1 time/hours time/from-now)
                          :iat   (time/now)}
                         (when sub
                               ;; when using the Admin API, delegating access, :sub may be needed
                               {:sub sub}))]
           (-> claim jwt/jwt (jwt/sign :RS256
                                       (-> creds
                                           :private_key
                                           (#(StringReader. %)) (#(key/pem->private-key % nil))))
               (jwt/to-str))))


(defn- request-token [scopes creds & [{:keys [sub] :as opts}]]
      (let [claim (create-claim scopes creds opts)
            resp (http/post "https://www.googleapis.com/oauth2/v4/token"
                            {:form-params {:grant_type "urn:ietf:params:oauth:grant-type:jwt-bearer"
                                           :assertion  claim}
                             :as          :json})]
           (when (= 200 (-> resp :status))
                 (println resp)
                 (-> resp :body :access_token))))


(defn get-bearer-token [scopes creds]
  "Get bearer token for requests"
      (let [rt (do (future (request-token scopes (load-creds creds))))]
           (reset! token @rt)))

