(ns ^:figwheel-always toshtogo.core
  (:require-macros [cljs.core.async.macros :refer [go-loop]])
  (:require [cljs.core.async :refer [chan <! put!]]
            [om.core :as om :include-macros true]
            [om.dom :as dom :include-macros true]
            [ajax.core :refer [GET]]
            [secretary.core :as secretary :refer-macros [defroute]]
            toastr

            [toshtogo.jobs.core :as jobs]

            [toshtogo.util.history :as history]
            [cemerick.url :as url]))

(enable-console-print!)

(defn notify
  [type msg]
  (case type
    :error (toastr/error msg)
    :success (toastr/success msg)))

(defn fetch-jobs
  [messages api-url]
  (println "FETCHING! " api-url)
  (GET api-url
       {:handler         (fn [response]
                           (put! messages [:jobs-fetched {:response response}]))

        :error-handler   (fn [response]
                           (put! messages [:failure {:response response}]))

        :keywords?       true

        :response-format :json}))

(defn fetch-job-types
  [messages]
  (GET "api/metadata/job_types"
       {:handler         (fn [response]
                           (put! messages [:job-types-fetched {:response response}]))

        :error-handler   (fn [response]
                           (put! messages [:failure {:response response}]))

        :keywords?       true

        :response-format :json}))

(defn build-routes
  [data messages]

  ; Figwheel-driven development requirement
  (secretary/reset-routes!)

  (secretary/add-route! "/" (fn [_]
                              (history/navigate (str "/jobs?source=" (url/url-encode "api/jobs?page=1&page_size=2")))))

  (secretary/add-route! "/jobs" (fn [{{:keys [source]} :query-params}]
                                  (om/transact! data #(assoc %
                                                       :status :loading))
                                  (fetch-jobs messages source)
                                  (fetch-job-types messages))))

(defn app-view
  [data owner]
  (reify
    om/IInitState
    (init-state [_]
      {:messages (chan)})

    om/IWillMount
    (will-mount [_]
      (aset toastr/options "positionClass" "toast-top-right")
      (let [messages (om/get-state owner :messages)]
        (build-routes data messages)
        (go-loop []
          (when-let [[event body] (<! messages)]
            (case event
              :jobs-fetched (let [{:keys [response]} body]
                              (println response)
                              (om/transact! data #(merge % {:status :done
                                                            :jobs   (:data response)
                                                            :paging (:paging response)})))

              :job-types-fetched (let [{:keys [response]} body]
                                   (om/transact! data #(merge % {:status    :done
                                                                 :job-types response})))
              :failure (do
                         (notify :error "Oh dear")
                         (om/transact! data #(assoc % :status :error :error-body (keyword (:error body)))))

              (throw (str "Unknown event: " event))))
          (recur))))

    om/IRenderState
    (render-state [_ state]
      (dom/div #js {:className "col-xs-12"}
        (dom/div #js {:className "col-xs-2"})
        (dom/div #js {:className "col-xs-9"}
          (om/build jobs/jobs-view data))
        ))))

(def app-state (atom {:search {:page-size 2}}))

(om/root
  app-view
  app-state
  {:target (. js/document (getElementById "app"))})

(defn on-js-reload []
  ;; optionally touch your app-state to force rerendering depending on
  ;; your application
  (swap! app-state update-in [:__figwheel_counter] inc)
  )
