(ns samsara.client
  (:require [taoensso.timbre :as log])
  (:require [org.httpkit.client :as http])
  (:require [validateur.validation :as val])
  (:require [clojure.data.json :as json])
  (:require [samsara.ring-buffer :refer :all])
  (:require [chime :refer [chime-ch]]
            [clj-time.core :as t]
            [clj-time.periodic :refer [periodic-seq]]
            [clojure.core.async :as a :refer [<! go-loop]]))

;; Config
;; Map containing the following config:
;; :url - Samsara url
;; :publish-interval - how often should events be flushed to the api
;; :max-buffer-size - Max size of the ring buffer.
;; :sourceId is autogenerated for each client. [TODO: Currently just a randomUUID. Make this persistent.]
(def ^{:private true} samsara-config
  (atom {:url "http://samsara.io/v1"
         :sourceId (str (java.util.UUID/randomUUID))
         :publish-interval 3600 ;seconds
         :max-buffer-size 1500
         }))

(defn get-samsara-config [] (deref samsara-config))

(defn set-config! [config]
  "Set samsara configuration.
   The following properties can be set:
   :url - Samsara URL
   :sourceId - Unique ID for the client. This property is defaulted with an autogenerated value.
   :publish_interval - Frequency in seconds, in which the events will be flushed to samsara API.
   :max_buffer_size - Max size of the events ring buffer.

   NOTE: Changes to publish_interval and max_buffer_size properties will require a restart immediately
   after the first call to record-event"
  (swap! samsara-config into config))


(def ^{:private true} event-validation-set

  (val/validation-set
   (val/presence-of :eventName :message "is required")
   (val/presence-of :timestamp :message "is required")
   (val/presence-of :sourceId :message "is required")
   (val/numericality-of :timestamp :message "should be numeric")))

(defn- validation-error-msg [errors]
  "Takes validation errors and returns a single line containing all errors separated by commas."
  (apply str  (interpose "," (flatten (for [err (seq errors)
                                            :let [key (first err)
                                                  msgs (second err)]]
                                        (map #(str key " " %1) msgs))))))

(defn- validate-event [event]
  "Validates the event and throws an Exception if Invalid"
  (let [errs (seq (event-validation-set event))]
    (if (nil? errs)
      true
      (throw (IllegalArgumentException. (validation-error-msg errs)))
      )))

(def ^{:private true} !event-headers! (atom {}))

(defn set-event-headers! [headers]
  "Set event headers"
  (swap! !event-headers! into headers))

(defn get-event-headers [] (deref !event-headers!))

(defn- enrich-event [event]
  "Enriches the event with default properties etc."
  (conj event (get-event-headers) {:timestamp (System/currentTimeMillis)
                                   :sourceId ((deref samsara-config) :sourceId)}))

(defn- prepare-event [event]
  "Enriches and validates the event and throws an Exception if validation fails."
  (let [e (enrich-event event)]
    (validate-event e)
    e))

(defn- send-events [events]
  "Send events to samsara api"
  (let [{:keys [status error] :as resp} @(http/post (str ((get-samsara-config) :url) "/events")
                                                    {:timeout 500 ;;ms
                                                     :headers {"Content-Type" "application/json"
                                                               "X-Samsara-publishedTimestamp" (str (System/currentTimeMillis))}
                                                     :body (json/write-str events)})]
    ;;Throw the exception from HttpKit to the caller.
    (when error
      (log/error "Failed to connect to samsara with error: " error)
      (throw error))
    ;;Throw an exception if status is not 201.
    (when (not (= 202 status))
      (log/error "Publish failed with status:" status)
      (throw (Exception. (str "PublishFailed with status=" status)))
      )
    )
  )

(defn publish-events [events]
  "Takes a vector containing events and publishes to samsara immediately."
  (let [e (map #(prepare-event %1) events)]
    (send-events e)))


(def ^:private !buffer! (atom nil))

(defn- flush-buffer []
  "Flushes the event buffer to samsara api. Does nothing if another flush-buffer is in progress."
  (try
    (let [events (seq (snapshot @!buffer!))]
      (if (not (nil? events))
        (try
          (send-events (map second events))
          (dequeue! @!buffer! events)
          (catch Throwable t (log/error t "Flush failed. Leaving events in the buffer to try again.")))
        (log/info "Nothing to send.")))))

(defn- timer-flush []
  (log/warn "Timer is flushing now")
  (flush-buffer))

(defn !init! []
  "Initializes the ring buffer and the timer."
  (let [config (get-samsara-config)
        max-buffer-size (config :max-buffer-size)
        did-set (compare-and-set! !buffer! nil (ring-buffer max-buffer-size))
        publish-interval (config :publish-interval)
        times (periodic-seq (t/now) (-> publish-interval t/seconds))]

    (when did-set
      (log/info "Starting job to flush events.")
      (let [chimes (chime-ch times {:ch (a/chan (a/sliding-buffer 1))})]
        (go-loop []
          (when-let [time (<! chimes)]
            (log/info "Flushing buffer now.")
            (timer-flush)
            (recur)))))))


(defn record-event [event]
  "Buffers the events to be published later."
  (!init!) ;;Init if not initialised already
  (enqueue! @!buffer! (prepare-event event)))
