(ns leiningen.version-suffix
  (:require
    [clojure.string :as str]
    [clojure.java.io :refer [reader]]
    [leiningen.core.main :refer [abort]]
    [me.raynes.fs :refer [copy]]
    [org.satta.glob :as clj-glob]
    )
  (:import
    (java.io File FileInputStream FileOutputStream)
    (java.util.zip GZIPOutputStream)
    (org.apache.commons.codec.digest DigestUtils)
    (org.apache.commons.codec.binary Base32)
    (java.util.regex Pattern)))

(def ^:dynamic *verbose* true)

(defn message [& args]
  (if *verbose* (apply println args)))

(defn validate-root [config]
  (let [root (:root config)]
    (if (nil? root)
      (abort ":root key under :version-suffix is required.")
      (if-not (string? root)
        (abort ":root key under :version-suffix must be a string.")
        (let [file (File. root)]
          (if-not (.exists file)
            (abort (format "Version suffix root %s does not exist." root))
            (if-not (.isDirectory file)
              (abort (format "Version suffix root %s is not a directory." root)))))))))

(defn validate-config [{:keys [assets references] :as config}]
  (if-not (map? config)
    (abort ":version-suffix key must be a map.")
    (do
      (validate-root config)
      (when-not (and (vector? assets) (vector? references))
        (abort "both :assets and :references keys under :version-suffix must be vectors.")))))

(defn path-exists [path]
  (.exists (File. path)))

(defn output-dir [project]
  (or (get-in project [:version-suffix :output-to])
      (first (filter path-exists (:resource-paths project)))
      (first (filter path-exists (:source-paths project)))))

(defn split-ext [name]
  (let [dotpos (.indexOf name ".")]
    (if (neg? dotpos)
      [name nil]
      [(subs name 0 dotpos) (subs name dotpos)])))

(defn name-with-version [name version]
  (let [[base ext] (split-ext name)]
    (str base "-" version ext)))

(defn target-file [^File original ^String version]
  (let [parent (.getParent original)
        name  (.getName original)
        output-name (name-with-version name version)]
    (File. parent output-name)))

(defn digest-file [^File file]
  (with-open [stream (FileInputStream. file)]
    (let [bytes (DigestUtils/sha1 stream)]
      (.toLowerCase (.encodeToString (Base32.) bytes)))))

(defn gzip-file [file]
  (let [gzip-file (File. (str (.getPath file) ".gz"))]
    (with-open [input (FileInputStream. file)
                output (GZIPOutputStream. (FileOutputStream. gzip-file))]
      (let [buffer (byte-array 8192)
            fill-buffer (fn [] (.read input buffer))]
        (loop [read-count (fill-buffer)]
          (when (pos? read-count)
            (.write output buffer 0 read-count)
            (recur (fill-buffer))))))
    (.setLastModified gzip-file (.lastModified file))))

(defn versioned-file-filter [file]
  (let [[base ext] (split-ext (.getName file))
        pattern (re-pattern (str (Pattern/quote base)
                                 "-\\w{32}"
                                 (Pattern/quote ext)
                                 "(?:\\.gz)?"))]
    (reify
      java.io.FilenameFilter
      (accept [_this _dir name]
        (.matches (re-matcher pattern name))))))

(defn clean-old-versions [file]
  (let [parent (.getParentFile file)
        files (.listFiles parent (versioned-file-filter file))]
    (doseq [file files]
      (.delete file)
      (message "Removed stale versioned file" (.getPath file)))))

(defn rename-file [config path]
  (let [root (str "./" (:root config))
        file (File. path)]
    (let [version (digest-file file)
          target (target-file file version)]
      (if (:clean config) (clean-old-versions file))
      (if (:copy? config)
        (copy file target)
        (.renameTo file target))
      (message "Versioned" (.getPath file) "->" (.getPath target))
      (if (:gzip config) (gzip-file target))
      [(subs (.getPath file) (.length root)) (subs (.getPath target) (.length root))])))

(defn expand-globs [config key]
  (let [root (:root config)
        globs (key config)]
    (->> (map #(clj-glob/glob (str root %)) globs)
         flatten
         (map #(.getPath %) ,,,))))

(defn version-references [config runtime-data]
  (let [root (:root config)
        references (expand-globs config :references)
        files (into {} (map (fn [path]
                              (let [contents (slurp path)]
                                [path (reduce (fn [contents [k v]]
                                                (let [[base ext] (split-ext k)
                                                      pattern (re-pattern (str (Pattern/quote base)
                                                                               "(-\\w{32})?"
                                                                               (Pattern/quote ext)
                                                                               "(?:\\.gz)?"))]
                                                  (str/replace contents pattern #(do (message "Replaced in" path ":" (first %)) v)))) 
                                              contents
                                              runtime-data)]))
                            references))]
    (doseq [[path contents] files] (spit path contents))))

(defn version-files [project]
  (let [config (:version-suffix project)
        assets (expand-globs config :assets)
        runtime-data (into {} (map (partial rename-file config) assets))
        output-to (File. (output-dir project) "assets.edn")]
    (spit output-to runtime-data)
    (version-references config runtime-data)
    runtime-data))

(defn version-suffix
  "Add a version suffix to files."
  [project & args]
  (let [config (:version-suffix project)]
    (when-not (nil? config)
      (validate-config config)
      (binding [*verbose* (not (:quiet config))]
        (version-files project)))))
