(ns clj-ftp-client.with-default-connection
  "Similar to clj-ftp-client.core, but has default connection hander."
  (:refer-clojure :exclude [get])
  (:require [clj-ftp-client.core :as core])
  (:import (org.apache.commons.net.ftp FTPClient)))

(def ^:dynamic *client*
  "The default ftp client handler"
  nil)

(defn set-client!
  "Sets the default connected ftp client handler."
  [client-OR-client-spec-map]
  (let [new-client (if (map? client-OR-client-spec-map)
                     (core/open client-OR-client-spec-map)
                     client-OR-client-spec-map)]
    (alter-var-root #'*client*
                    (constantly new-client)
                    (when (thread-bound? #'*client*)
                      (set! *client* new-client)))))

(defmacro with-client
  "Establish an FTP connection, bound to *client*, and execute the body.
  Closes connection at the end of `body`.

  client-spec-map is equivalent to `clj-ftp-client.core/open`'s argument."
  [client-spec-map & body]
  `(with-open [client# ^FTPClient (core/open ~client-spec-map)]
     (binding [*client* client#]
       ~@body)))


(defmacro ^:private extend-function [name]
  (let [core-sym (symbol "clj-ftp-client.core" (str name))
        {:keys [doc arglists]} (meta (resolve core-sym))
        doc (str doc
                 "\n\n  Uses default ftp client hander if `client` is not given.")]
    `(defn ~(with-meta name (assoc (meta name) :doc doc))
       ~@(map (fn [args]
                `(~args (~core-sym ~@args)))
              arglists)
       ~@(map (fn [args]
                `(~args (~core-sym *client* ~@args)))
              (map (comp vec rest) arglists)))))
(extend-function ls)
(extend-function cd)
(extend-function pwd)
(extend-function mkdir)
(extend-function mkdirs)
(extend-function rm)
(extend-function rmdir)
(extend-function mv)

(defmacro ^:private extend-with-x-file-stream [name]
  (let [core-sym (symbol "clj-ftp-client.core" (str name))
        core-meta (-> core-sym resolve meta)
        doc (str (:doc core-meta)
                 "\n\n  Uses default ftp client hander if `client?` is not given.")
        arglists (->> (:arglists core-meta)
                      (map (fn [[params & rst]]
                             (vec (cons (->> params
                                             (map #(if (= 'client %) 'client? %))
                                             vec)
                                        rst)))))]
    `(do
       (defmacro ~name
         [[stream# remote-file-name# client?#] & body#]
         (let [client# (if client?# client?# `*client*)
               with-x-file-stream# '~core-sym]
           `(~with-x-file-stream# [~stream# ~remote-file-name# ~client#]
             ~@body#)))
       #_(alter-meta! ~(resolve (symbol "clj-ftp-client.with-default-connection" (str name)))
                    #(assoc %
                            :doc ~doc
                            :arglists '~arglists))
       ~(resolve (symbol "clj-ftp-client.with-default-connection" (str name))))))
(extend-with-x-file-stream with-retrieve-file-stream)
(extend-with-x-file-stream with-store-file-stream)
(extend-with-x-file-stream with-append-file-stream)


(defn ^{:doc (str (:doc (meta #'core/get))
                  "\n\n  Uses default ftp client hander if `client` is not given.")}
  get
  ([^String remote-file-name]
   (core/get *client* remote-file-name))
  ([^String remote-file-name local-file]
   (core/get *client* remote-file-name local-file))
  ([client ^String remote-file-name local-file]
   (core/get client remote-file-name local-file)))

(defn ^{:doc (str (:doc (meta #'core/put))
                  "\n\n  Uses default ftp client hander if `client` is not given.")}
  put
  ([local-file]
   (core/put *client* local-file))
  ([local-file ^String remote-file-name]
   (core/put *client* local-file remote-file-name))
  ([client local-file ^String remote-file-name]
   (core/put client local-file remote-file-name)))


(defmacro ^{:doc (:doc (meta #'core/with-rewinding-directory))}
  with-rewinding-directory
  [[client?] & body]
  (let [client (if client? client? `*client*)]
    `(core/with-rewinding-directory ~client
       ~@body)))
