;   Copyright (c) Ryan Wilson. All rights reserved.
;   The use and distribution terms for this software are covered by the
;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;   which can be found in the file epl-v10.html at the root of this distribution.
;   By using this software in any fashion, you are agreeing to be bound by
;   the terms of this license.
;   You must not remove this notice, or any other, from this software.

(ns clj-lib.perf
  "Useful utility fns for profiling code"
  (:import [java.lang System]))

(defn profile*
  "Executes f the specified number of times, returning profiling information
  at the end."
  ([f]
   {:pre [(ifn? f)]}
   (let [start (System/currentTimeMillis)]
     (f)
     (- (System/currentTimeMillis) start)))
  ([n f]
   {:pre [(ifn? f) (pos? n)]}
   (let [results (repeatedly n (partial profile* f))]
     {:min (apply min results)
      :max (apply max results)
      :total (apply + results)
      :avg (/ (apply + results) n)})))

(defmacro profile
  "Executes the specified body a specified number of times, printing profiling
   information at the end."
  [count & body]
  `(profile* ~count (fn [] ~@body)))
