;   Copyright (c) Ryan Wilson. All rights reserved.
;   The use and distribution terms for this software are covered by the
;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;   which can be found in the file epl-v10.html at the root of this distribution.
;   By using this software in any fashion, you are agreeing to be bound by
;   the terms of this license.
;   You must not remove this notice, or any other, from this software.

(ns clj-lib.io
  "Useful utility fns for local & network IO"
  (:import [java.io File BufferedReader InputStreamReader
                    ByteArrayInputStream ByteArrayOutputStream]
           [java.net URLEncoder URLDecoder]
           [java.util.zip Deflater Inflater DeflaterOutputStream InflaterOutputStream]
           [java.util Base64])
  (:require [clojure.java.io :as clj-io]
            [clojure.string :as clj-str]
            [clj-lib.types :refer (byte-array? to-string)]))

(def ^{:dynamic true
       :doc "Defines the default character encoding for fns in this namespace that
             require one. Specifically, string to/from bytes and URL encode/decode."}
  *char-encoding* "UTF-8")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; string to/from URL encoding
;;

(def url-encode #(URLEncoder/encode (str %) *char-encoding*))
(def url-decode #(URLDecoder/decode (str %) *char-encoding*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Serializing data to / from strings
;;

(defn serialize-string [s] (with-out-str (pr s)))
(defn deserialize-string [s] (read-string s))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; string to/from byte[]
;;

(defn str-to-bytes [s]
  {:pre [(string? s)]}
  (.getBytes (to-string s) *char-encoding*))

(defn bytes-to-str [bytes]
  {:pre [(byte-array? bytes)]}
  (String. bytes *char-encoding*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; byte[] to/from base64 encoding
;;

(def ^{:private true
       :doc "Base64.Encoder is thread-safe; this Var caches the instance so we
             only ever create one. Using delay defers instantiation until the
             instance is actually needed."}
  base64-encoder (delay (Base64/getEncoder)))

(def ^{:private true
       :doc "Base64.Decoder is thread-safe; this Var caches the instance so we
             only ever create one. Using delay defers instantiation until the
             instance is actually needed."}
  base64-decoder (delay (Base64/getDecoder)))

(defn base64-encode [b]
  {:pre [(byte-array? b)]}
  (.encode @base64-encoder ^bytes b))

(defn base64-encode-to-str [b]
  {:pre [(byte-array? b)]}
  (.encodeToString @base64-encoder ^bytes b))

(defn base64-decode [v]
  "accepts either a string or a byte-array, base64 decodes the value,
   and returns a byte-array."
  (cond
    (string? v) (.decode @base64-decoder ^String v)
    (byte-array? v) (.decode @base64-decoder ^bytes v)
    :else (throw (IllegalArgumentException. (str "Cannot base64-decode type: "
                                                 (type v))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Compression & Decompression using zlib
;;

(def ^{:dynamic true
       :doc "Defines the size of the compress/decompress stream buffers, in bytes"}
  *zip-buff-size* 1024)

(defn compress
  "Compresses the input string and returns it as a byte array, using zlib compression"
  [s]
  (let [bytes (str-to-bytes s)
        out (ByteArrayOutputStream.)]
    (with-open [compresser (DeflaterOutputStream. out (Deflater.) *zip-buff-size*)]
      (.write compresser ^bytes bytes))
    (.toByteArray out)))

(defn decompress
  "Decompresses a zlib compressed byte-array, returns the value as a string"
  [b]
  {:pre [(byte-array? b)]}
  (let [out (ByteArrayOutputStream.)]
    (with-open [decompresser (InflaterOutputStream. out (Inflater.) *zip-buff-size*)]
      (.write decompresser ^bytes b))
    (bytes-to-str (.toByteArray out))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Very simple HTTP ops
;;

(defn curl
  "Simple URL fetcher al la bash curl (but without the options!) and returns it as a
   string, which may be quite inefficient depending on the size of the content."
  [url]
  {:pre [(some? url)]}
  (with-open [stream (.openStream (clj-io/as-url url))]
    (let [buf (BufferedReader. (InputStreamReader. stream))]
      (clj-str/join (line-seq buf)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; File Utils
;;

(defn ensure-dirs
  "Ensures that all directories in the specified `path` exist, creating them if
  necessary."
  [path]
  (let [parent-dir (.getParentFile (clj-io/as-file path))]
    (when-not (.exists parent-dir)
      (.mkdirs parent-dir)))
)
