(ns common.db.ops
  "@ctdean"
  (:refer-clojure :exclude [read update])
  (:require
    [clojure.java.jdbc :as jdbc]
    [common.db.pool :refer [datasource]]
    [common.db.util :refer [now]]))

(defn create [table tspec obj]
  (let [t (now)
        new (-> obj
                (assoc :updated_at t
                       :created_at t))
        dbspec (or tspec datasource)]
    (first (jdbc/insert! dbspec table new))))

(defn read [table tspec id]
  (first (jdbc/query (or tspec datasource)
                     [(str "select * from " (name table) " where id = ?") id])))

(defn read-all [table tspec]
  (jdbc/query (or tspec datasource) (str "select * from " (name table))))

(defn update [table tspec obj]
  (assert (:id obj) "id required")
  (let [new (-> obj
                (assoc :updated_at (now))
                (dissoc :id :created_at))
        nupdated (jdbc/update! (or tspec datasource) table new ["id = ?" (:id obj)])]
    (> (first nupdated) 0)))

(defn delete [table tspec id]
  (assert id "id required")
  (let [dbspec (or tspec datasource)
        ndel (jdbc/delete! dbspec table ["id = ?" id])]
    (> (first ndel) 0)))

(defmacro define-crud-ops
  ([table]
   `(define-crud-ops ~table nil))
  ([table prefix]
   `(do
      (defn ~(symbol (str prefix "create"))
        ([obj#] (create ~table nil obj#))
        ([spec# obj#] (create ~table spec# obj#)))
      (defn ~(symbol (str prefix "read"))
        ([id#] (read ~table nil id#))
        ([spec# id#] (read ~table spec# id#)))
      (defn ~(symbol (str prefix "read-all"))
        ([] (read-all ~table nil))
        ([spec#] (read-all ~table spec#)))
      (defn ~(symbol (str prefix "update"))
        ([obj#] (update ~table nil obj#))
        ([spec# obj#] (update ~table spec# obj#)))
      (defn ~(symbol (str prefix "delete"))
        ([id#] (delete ~table nil id#))
        ([spec# id#] (delete ~table spec# id#))))))
