(ns leiningen.give-me-my-css
  (:require [clojure.java.io :as io]
            [clojure.java.shell :as shell]
            [clojure.string :as s])
  (:import java.lang.ProcessBuilder))

(def ^:const docker-image "ubuntudesign/sass")

(defn inside-docker? []
  (.exists (io/file "/.dockerenv")))

(defn container-id []
  (->> "cat /proc/1/cgroup | grep 'docker/' | tail -1 | sed 's/^.*\\///' | xargs echo -n"
       (shell/sh "bash" "-c")
       :out))

(defn sass [{:keys [input output mode style source-map no-cache]
             :or   {input      "resources/public/scss"
                    output     "resources/public/css"
                    mode       :once
                    style      :nested
                    source-map true
                    no-cache   false}
             :as   args}]
  (println "building sass/scss with options:" (pr-str {:input      input
                                                       :output     output
                                                       :mode       mode
                                                       :style      style
                                                       :source-map source-map
                                                       :no-cache   no-cache}))
  (let [dir            (.getAbsolutePath (io/file ""))
        mount-dir      (if (inside-docker?)
                         dir
                         "/app")
        docker-command (cond-> ["docker" "run" "--rm"]
                         (not (inside-docker?)) (conj "-v" (format "%s:%s" dir mount-dir))
                         (inside-docker?)       (conj "--volumes-from" (container-id)
                                                      "-w" dir))
        docker-command (conj docker-command docker-image)
        mode-arg       (case mode
                         "watch" "--watch"
                         "auto"  "--watch"
                         "--update")
        style          (if style
                         (name style)
                         "nested")
        args           (cond-> (conj docker-command
                                     "sass" mode-arg
                                     (format "%s/%s:%s/%s" mount-dir input mount-dir output)
                                     "--style" style)
                         source-map       (conj "--sourcemap")
                         no-cache         (conj "-C")
                         (= "force" mode) (conj "-f"))
        _              (apply println "running:" args)
        pb             (ProcessBuilder. args)
        _              (.inheritIO pb)
        _              (.directory pb (io/file dir))
        proc           (.start pb)
        _              (.waitFor proc)
        exit-code      (.exitValue proc)]
    (println "build finished")
    (when (pos? exit-code)
      (System/exit exit-code))))

(defn give-me-my-css
  [{:keys [give-me-my-css]} & [mode]]
  (if (= "help" mode)
    (do
      (println "Usage:")
      (println "lein give-me-my-css or lein give-me-my-css once -- to compile sass once")
      (println "lein give-me-my-css force -- to force sass recompilation")
      (println "lein give-me-my-css watch or lein give-me-my-css auto -- to start watcher")
      (println "lein give-me-my-css help -- to show this output"))
    (sass (cond-> give-me-my-css
            mode (assoc :mode mode)))))
