Baum
====

**Baum** is an extensible configuration library for Clojure.

Baum is built on top of `clojure.tools.reader.edn` and offers following
features with extended EDN notation.

-   referring to envirionment variables
-   including other files
-   binding subtree to a symbol and refer it later
-   environment specific config
-   etc

Actually Baum does not extend EDN syntax itself. It achives these
features by reader macros and post-parsing reduction.

Setup
-----

Add the following dependency in your `project.clj` file:

``` {.clojure}
  [rkworks/baum "0.1.0-SNAPSHOT"]
```

Reading your config file
------------------------

To read your config files, use `read-config`:

``` {.clojure}
  (ns user
    (:require [baum.core :as b]))

  (def config (b/read-config "path/to/config.edn")) ; a map

  (:foo config)
  (get-in config [:foo :bar])
```

It also supports other arguments:

``` {.clojure}
  (ns user
    (:require [baum.core :as b]
              [clojure.java.io :as io]))

  (def config (b/read-config (io/resource "config.edn")))

  (def config2 (b/read-config (java.io.StringReader. "{:a :b}"))) ; Same as (b/read-string "{:a :b}")
```

See `clojure.java.io/reader` for a complete list of supported arguments.

Built-in Reader Macros
----------------------

### baum/env

Read environment variables:

``` {.clojure}
  {:foo #baum/env :user}                  ; => {:foo "rkworks"}
```

[Environ](https://github.com/weavejester/environ) is used internally. So
you can also read Java properties, a `.lein-env` file, or your
`project.clj` (you need `lein-env` plugin). For more details, see
Environ's README.

You can also set fallback value:

``` {.clojure}
  #baum/env [:non-existent-env "not-found"]       ; => "not-found"
  #baum/env [:non-existent-env :user "not-found"] ; => "rkworks"
  #baum/env ["foo"]                               ; => "foo"
  #baum/env []                                    ; => nil

  ;; real-world example
  {:port #baum/env [:port 3000]}
```

### baum/cond

You can write conditional configurations with `baum/cond`.

``` {.clojure}
  {:database
   #baum/cond [#baum/env :env
               {"prod" {:host     "xxxx"
                        :user     "root"
                        :password "aaa"}
                "dev"  {:host     "localhost"
                        :user     "root"
                        :password "bbb"}
                :else  {:host     "localhost"
                        :user     "root"
                        :password nil}}]}
```

`baum/cond` accepts a vector and transforms it according to the value of
its first element. In the above example, if `#baum/env :env` is "prod",
the result is:

``` {.clojure}
  {:database {:host     "xxxx"
              :user     "root"
              :password "aaa"}}
```

If the value is neither "prod" nor "dev", the result is:

``` {.clojure}
  {:database {:host     "localhost"
              :user     "root"
              :password nil}}
```

### baum/env-cond

`baum/env-cond` is syntactic sugar for combination use of `baum/cond`
and `baum/env`.

The following example is `baum/env-cond` version of the above example.

``` {.clojure}
  {:database
   #baum/env-cond [:env
                   {"prod" {:host     "xxxx"
                            :user     "root"
                            :password "aaa"}
                    "dev"  {:host     "localhost"
                            :user     "root"
                            :password "bbb"}
                    :else  {:host     "localhost"
                            :user     "root"
                            :password nil}}]}
```

### baum/file

To embed File objects in your configuration files, you can use
`baum/file`:

``` {.clojure}
  {:file #baum/file "project.clj"}      ; => {:file #<File project.clj>}
```

### baum/resource

Your can also refer resouce files via `baum/resource`:

``` {.clojure}
  {:resource #baum/resource "config.edn"}
  ;; => {:resource #<URL file:/path/to/project/resources/config.edn>}
```

### baum/import

You can use `baum/import` to import config from other files.

child.edn:

``` {.clojure}
  {:child-key :child-val}
```

parent.edn:

``` {.clojure}
  {:parent-key #baum/import "path/to/child.edn"}
  ;; => {:parent-key {:child-key :child-val}}
```

If you want to import a resouce file, use `baum/resource` together:

``` {.clojure}
  {:a #baum/import #baum/resource "config.edn"}
```

**NB:** The reader throws an exception if you try to import a non
existent file.

### baum/import\*

Same as `baum/import`, but returns nil when error occurs:

``` {.clojure}
  {:a #baum/import* "non-existent-config.edn"} ; => {:a nil}
```

### baum/some

`baum/some` returns the first logical true value of a given vector:

``` {.clojure}
  #baum/some [nil nil 1 nil]              ; => 1

  #baum/some [#baum/env :non-existent-env
              #baum/env :user]            ; => "rkworks"

```

Let's see the next example. In this case, if `~/.private-conf.clj`
exists, the result is its content, otherwise `{:not-found true}`

``` {.clojure}
  #baum/some [#baum/import* "~/.private-conf.clj"
              {:not-found true}]
```

### baum/eval

To embed Clojure code in your configuration files, use `baum/eval`:

``` {.clojure}
  {:timeout #baum/eval (* 1000 60 60 24 7)} ; => {:timeout 604800000}
```

### baum/ref

You can refer bound variables with `baum/ref`. For more details, see the
explanation of **:baum/let**.

Bult-in Reducers
----------------

### :baum/include

`:baum/include` key merges deeply its child with its owner map.

For example:

``` {.clojure}
  {:baum/include {:a :child}
   :a :parent}                        ; => {:a :parent}
```

In the above example, a reducer merges `{:a :parent}` into
`{:a :child}`.

`:baum/include` also accepts a vector:

``` {.clojure}
  {:baum/include [{:a :child1} {:a :child2}]
   :b :parent}                            ; => {:a :child2 :b :parent}
```

In this case, the merging strategy is like the following:

``` {.clojure}
  (deep-merge {:a :child1} {:a :child2} {:b :parent})
```

Finally, it accepts all other importable values.

For example:

``` {.clojure}
  ;; child.edn
  {:a :child
   :b :child}

  ;; config.edn
  {:baum/include "path/to/child.edn"
   :b :parent}                            ; => {:a :child :b :parent}
```

Of course it is possible to pass a vector of importable values:

``` {.clojure}
  {:baum/include ["child.edn"
                  #baum/resource "resource.edn"]
   :b :parent}
```

### :baum/include\*

Same as `:baum/include`, but ignores importing errors:

``` {.clojure}
  ;; child.edn
  {:foo :bar}

  ;; config.edn
  {:baum/include* ["non-existent-file.edn" "child.edn"]
   :parent :qux}                          ; => {:foo :bar :parent :qux}
```

It is equivalent to the following operation:

``` {.clojure}
  (deep-merge nil {:foo :bar} {:parent :qux})
```

### :baum/override

The only difference between `:baum/override` and `:baum/include` is the
merging strategy. In contrast to `:baum/include`, `:baum/override`
merges child values into a parent map.

In the next example, a reducer merges `{:a :child}` into `{:a :parent}`.

``` {.clojure}
  {:baum/override {:a :child}
   :a :parent}                            ; => {:a :child}
```

### :baum/override\*

Same as `:baum/override`, but ignores importing errors. See also
`:baum/include*`.

### :baum/let

You can use `:baum/let` and `baum/ref` to make a part of your config
reusable:

``` {.clojure}
  {:baum/let [a 100]
   :a #baum/ref a
   :b {:c #baum/ref a}}            ; => {:a 100 :b {:c 100}}
```

Destructuring is available:

``` {.clojure}
  {:baum/let [{:keys [a b]}  {:a 100 :b 200}]
            :a #baum/ref a
            :b #baum/ref b}
  ;; => {:a 100 :b 200}

  {:baum/let [[a b] [100 200]]
   :a #baum/ref a
   :b #baum/ref b}
  ;; => {:a 100 :b 200}
```

Of course, you can use other reader macros together:

``` {.clojure}
  ;;; a.edn
  {:foo :bar :baz :qux}

  ;;; config.edn
  {:baum/let [{:keys [foo baz]} #baum/import "a.edn"]
   :a #baum/ref foo
   :b #baum/ref baz}
  ;; => {:a :bar :b :qux}
```

`baum/let`'s scope is determined by hierarchical structure of config
maps:

``` {.clojure}
  {:baum/let [a :a
              b :b]
   :d1 {:baum/let [a :d1-a
                   c :d1-c]
        :a #baum/ref a
        :b #baum/ref b
        :c #baum/ref c}
   :a #baum/ref a
   :b #baum/ref b}
  ;; => {:d1 {:a :d1-a
  ;;          :b :b
  ;;          :c :d1-c}
  ;;     :a  :a
  ;;     :b  :b}
```

You will get an error if you try to access an unavailable variable:

``` {.clojure}
  {:a #baum/ref a
   :b {:baum/let [a 100]}}
  ;; => Error: "Unable to resolve symbol: a in this context"
```

Writing your own reader macros
------------------------------

It is very easy to write reader macros. To write your own, use
`defreader`.

config.edn:

``` {.clojure}
  {:foo #greet "World"}
```

your~ns~.clj:

``` {.clojure}
  (ns your-ns
    (:require [baum.core :as b]))

  (b/defreader greeting-reader [v opts]
    (str "Hello, " v "!"))

  ;; Put your reader macro in reader options:
  (b/read-config "config.edn"
                 {:readers {'greet greeting-reader}}) ; => {:foo "Hello, World!"}

  ;; Another way to enable your macro:
  (binding [*data-readers* (merge *data-readers*
                                  {'greet greeting-reader})]
    (b/read-config "config.edn"))
```

For more complex examples, see implementations of built-in readers.

### Differences from Clojure's reader macro definition

If you have ever written reader macros, you may wonder why you should
use `defreader` to define them even though they are simple unary
functions.

This is because it is necessary to synchronize the evaluation timing of
reducers and reader macros. To achive this, `defreader` expands a
definition of a reader macro like the following:

``` {.clojure}
  (defreader greeting-reader [v opts]
    (str "Hello, " v "!"))

  ;;; ↓↓↓↓↓↓↓↓↓↓↓↓↓↓↓↓↓

  (let [f (fn [v opts]
            (str "Hello, " v "!"))]
    (defn greeting-reader [v]
      {:baum.core/invoke [f v]}))
```

So, the actual evaluation timing of your implementation is the reduction
phase and this is performed by an internal built-in reducer.

One more thing, you can access reader options!

Writing your own reducers
-------------------------

In contrast to reader macros, there is no macro to define reducers. All
you need to do is define a ternary function. Consider the following
reducer:

``` {.clojure}
  {:your-ns/narrow [:a :c]
   :a :foo
   :b :bar
   :c :baz
   :d :qux}

  ;;; ↓↓↓↓↓↓↓↓↓↓↓↓↓↓↓↓↓

  {:a :foo
   :c :baz}
```

To implement this, you colud write the following:

``` {.clojure}
  (ns your-ns
    (:require [baum.core :as b]))

  (defn narrow [v m opts]
    (select-keys m v))

  ;; Put your reducer in reader options:
  (b/read-config "config.edn"
                 {:reducers {:your-ns/narrow greeting-reader}})
```

In the above example, `v` is a value under the `:your-ns/narrow` key and
`m` is a map from which the `:your-ns/narrow` key has been removed.
`opts` is reader options. So `narrow` will be called as follows:

``` {.clojure}
  (narrow [:a :c]
          {:a :foo :b :bar :c :baz :d :qux}
          {...})
```

License
-------

Copyright © 2015 Ryo Fukumuro

Distributed under the Eclipse Public License, the same as Clojure.
