(ns {{name}}.init
  (:require [clojure.edn :as edn]
            [com.stuartsierra.component :as component]
            [org.httpkit.server :as http-server :refer [run-server]]
            reverie.nsloader
            [reverie.admin :as admin]
            [reverie.admin.api.editors :refer [get-edits-task]]
            [reverie.cache :as cache]
            [reverie.cache.memory :as cache.memory]
            [reverie.cache.sql :as cache.sql]
            [reverie.database.sql :as db.sql]
            ;; [reverie.email :as email]
            [reverie.i18n :as i18n]
            [reverie.logger :as logger]
            [reverie.migrator :as migrator]
            [reverie.migrator.sql :as migrator.sql]
            [reverie.modules.filemanager :as fm]
            [reverie.modules.role :as rm]
            [reverie.page :as page]
            [reverie.scheduler :as scheduler]
            [reverie.server :as server]
            [reverie.settings :as settings]
            [reverie.site :as site]
            [reverie.system :refer [load-views-ns
                                    load-views] :as sys]
            [taoensso.timbre :as log]))


(defn- system-map [{:keys [prod? logger-settings db-specs ds-specs settings
                           host-names render-fn
                           ;; email-settings
                           base-dir media-dirs
                           cache-store site-hash-key-strategy
                           server-options middleware-options
                           i18n-tconfig
                           run-server stop-server]}]
  (let [logger (component/start (logger/logger prod? logger-settings))
        db (component/start (db.sql/database (not prod?) db-specs ds-specs))]
    ;; run the migrations
    (->> db
         (migrator.sql/get-migrator)
         (migrator/migrate))

    (component/system-map
     :database db
     :settings settings
     :rolemanager (component/using (rm/get-rolemanager)
                                   [:database])
     :i18n (component/using (i18n/get-i18n prod? i18n-tconfig) [])
     :server (component/using (server/get-server {:server-options server-options
                                                  :run-server run-server
                                                  :stop-server stop-server
                                                  :middleware-options middleware-options
                                                  :dev? (not prod?)})
                              [:filemanager :site :database])
     :cachemanager (component/using
                    (cache/cachemananger {:store cache-store})
                    [:database])
     :filemanager (fm/get-filemanager base-dir media-dirs)
     :site (component/using (site/site {:host-names host-names
                                        :render-fn render-fn})
                            [:database :cachemanager])
     :logger logger
     :scheduler (scheduler/get-scheduler)
     ;; :email-manager (component/using
     ;;                 (email/email-manager email-settings) [])
     :admin (component/using (admin/get-admin-initializer)
                             [:database])
     :system (component/using (sys/get-system)
                              [:database :filemanager :site :scheduler
                               :settings :server :logger
                               :admin :cachemanager :i18n]))))


(defonce system (atom nil))

(defn stop []
  (when-not (nil? @system)
    (component/stop @system)
    (reset! system nil)))

(defn- stop-server [server]
  (when (fn? server)
    (server)))

(defn init [settings-path]
  ;; read in the settings first
  (let [settings (component/start (settings/settings settings-path))]

    ;; load namespaces after the system starts up
    ;; this step will set up any necessary migrations
    (load-views-ns '{{name}}.templates
                   '{{name}}.objects
                   '{{name}}.apps
                   '{{name}}.endpoints)

    ;; start the system
    (reset! system (component/start
                    (system-map
                     {:prod? (settings/prod? settings)
                      :logger-settings (settings/get settings [:log])
                      :settings settings
                      :i18n-tconfig (settings/get settings [:i18n :tconfig]
                                                  {:dictionary {}
                                                   :dev-mode? (settings/dev? settings)
                                                   :fallback-locale :en})
                      ;; :email-settings (settings/get settings [:email])
                      :db-specs (settings/get settings [:db :specs])
                      :ds-specs (settings/get settings [:db :ds-specs])
                      :server-options (settings/get settings [:server :options])
                      :middleware-options (settings/get settings [:server :middleware])
                      :run-server run-server
                      :stop-server stop-server
                      :host-names (settings/get settings [:site :host-names])
                      :render-fn hiccup.compiler/render-html
                      :base-dir (settings/get settings [:filemanager :base-dir])
                      :media-dirs (settings/get settings [:filemanager :media-dirs])
                      :cache-store (cache.memory/mem-store)})))

    ;; load the translations for i18n
    (->> @system
         :i18n
         (i18n/load-i18n!))

    ;; start up the scheduler with tasks
    (let [scheduler (-> @system :scheduler)
          cachemanager (-> @system :cachemanager)]
      (scheduler/add-tasks!
       scheduler
       [(get-edits-task
         (settings/get settings [:admin :tasks :edits :minutes]))
        (cache/get-prune-task
         (cache.sql/get-basicstrategy) cachemanager {})])
      (scheduler/start! scheduler))

    ;; log uncaught exceptions in threads
    (Thread/setDefaultUncaughtExceptionHandler
     (reify Thread$UncaughtExceptionHandler
       (uncaughtException [_ thread ex]
         (log/error {:what :uncaught-exception
                     :exception ex
                     :where (str "Uncaught exception on" (.getName thread))}))))

    ;; shut down the system if something like ctrl-c is pressed
    (.addShutdownHook
     (Runtime/getRuntime)
     (proxy [Thread] []
       (run []
         (stop))))))
