(ns leiningen.new.restapp
  (:require [leiningen.new.templates :refer [renderer name-to-path ->files]]
            [leiningen.core.main :as main]
            [clojure.string :as s]
            [clojure.set :refer [subset?]]))

(def render (renderer "restapp"))

(defn pad [len]
  (apply str (repeat len " ")))

(defn template-data [name opts]
  {:name           name
   :name-camel     (str (s/upper-case (subs name 0 1)) (subs name 1))
   :sanitized      (name-to-path name)
   :gen-rest-deps  (if (some #{:rest} opts)
                     (s/join (concat "\n" (pad 17) "[metosin/compojure-api \"2.0.0-alpha25\"]"
                                     "\n" (pad 17) "[ring-cors \"0.1.12\"]")))
   :gen-rest-handler (if (some #{:rest} opts)
                       (s/join (concat ":ring {:handler " name ".handler/app,\n"
                                       (pad 9) ":init " name ".handler/init\n"
                                       (pad 9) ":destroy " name ".handler/destroy}\n")))
   :gen-mysql-deps  (if (some #{:mysql} opts)
                      (s/join (concat "\n" (pad 17) "[mysql/mysql-connector-java \"8.0.12\"]")))
   :gen-oracle-deps  (if (some #{:oracle} opts)
                       (s/join (concat "\n" (pad 17) "[org.tovictory/oracle-adapter \"0.1.0-SNAPSHOT\"]")))
   :gen-postgres-deps  (if (some #{:postgres} opts)
                         (s/join (concat "\n" (pad 17) "[org.tovictory/postgresql-adapter \"0.1.0-SNAPSHOT\"]")))
   :gen-dev-plugins (if (some #{:rest} opts) (str "[lein-ring \"0.12.4\"]"))
   :gen-init-ns (if (some #{:rest} opts) (str name ".handler") (str name ".core"))})

(defmulti option-files (fn [option data] option))

(defmethod option-files :base [_ data]
  [["project.clj"                    (render "project.clj" data)]
   ["README.md"                      (render "README.md" data)]])

(defmethod option-files :rest [_ data]
  [["src/clj/{{sanitized}}/handler.clj"  (render "handler.clj" data)]
   ["src/clj/{{sanitized}}/middleware.clj"  (render "middleware.clj" data)]
   ["src/clj/{{sanitized}}/routes/demo.clj"  (render "demo-routes.clj" data)]
   ["src/clj/{{sanitized}}/services/demo.clj"  (render "demo-services.clj" data)]
   ["env/dev/clj/user.clj" (render "user.clj" data)]])

(defmethod option-files :db-base [_ data]
  [["doc/das/soil.html" (render "soil.html" data)]
   ["src/clj/{{sanitized}}/db/queries.clj" (render "db_core.clj" data)]
   ["resources/sql/queries.sql" (render "queries.sql" data)]])

(defmethod option-files :oracle [_ data]
  [["env/dev/resources/config.edn" (render "oracle-config.edn" data)]
   ["env/prod/resources/config.edn" (render "oracle-config.edn" data)]])

(defmethod option-files :mysql [_ data]
  [["env/dev/resources/config.edn" (render "mysql-config.edn" data)]
   ["env/prod/resources/config.edn" (render "mysql-config.edn" data)]])

(defmethod option-files :postgres [_ data]
  [["env/dev/resources/config.edn" (render "postgresql-config.edn" data)]
   ["env/prod/resources/config.edn" (render "postgresql-config.edn" data)]])

(defn active-options [args]
  (for [arg args :when (re-matches #"\+[A-Za-z0-9-]+" arg)]
    (keyword (subs arg 1))))

(defn restapp
  [name & args]
  (let [opts (cons :base (active-options args))
        opts (cons :rest opts)
        opts (if (some #{:oracle :mysql :postgres} opts) (cons :db-base opts) opts)
        data (template-data name opts)
        files (reduce into [] (map #(option-files % data) opts))]
    (main/info "Generating fresh restapp project.")
    (apply ->files data files)))
