(ns reify.tokamak.schemata
  (:require
    [schema.core :as s :include-macros true]
    [reify.tokamak.protocols :as p]
    #?(:clj [clojure.core.async.impl.protocols]
       :cljs [cljs.core.async.impl.protocols])))

(s/defschema State
  "This schema represents the (unknown) state. Tokamak cannot validate the
   schema itself and instead provides facilities, through `Machine`s, to do so
   in the specification of the application itself. Nevertheless, the `State`
   schema is used to denote the Tokamak-tracked application state used
   throughout a reactor."
  (s/named s/Any "application state"))

(s/defschema Action
  "A Tokamak Flux action (or 'event') is a map of a specific form modeled off
  `acdlite/flux-standard-action`.

  The `type` value specifies the type of the action indicating a (subtyping)
  lattice of specific action types where `[]` is the TOP type.

  The `payload` value qualifies the action and should be typed according to
  the particular type of action being handled.

  The `error` value (as a departure from FSA format) exists iff the action
  signifies an error occurred and in that case is some type of error
  descriptor, specific to the type of action.

  This action specification differs again from FSA by not including a `meta`
  value, but this is, of course, provided naturally through Clojure's own
  metadata facilities."
  {:type                     [s/Keyword]
   (s/optional-key :payload) s/Any
   (s/optional-key :error)   s/Any})

(s/defschema Reducer
  "A Tokamak reducer is, semantically, a (pure) function from application
   state and an action to updated application state, `(e, s) -> s`. In practice,
   Tokamak also accepts maps from action type specializers to reducers to enable
   both efficient description of reducers and efficient dispatch."
  (s/=> State Action State))

(s/defschema Channel
  "A core.async channel."
  (s/protocol #?(:clj  clojure.core.async.impl.protocols.Channel
                 :cljs cljs.core.async.impl.protocols.Channel)))

(s/defschema Ratom
  "A ratom is a reagent concept of a reactive atom."
  (s/protocol #?(:cljs IDeref :clj clojure.lang.IDeref)))

(s/defschema Reaction
  "A reaction is a reagent concept of a reactive computation."
  (s/protocol #?(:cljs IDeref :clj clojure.lang.IDeref)))

(s/defschema Nil (s/eq nil))

(s/defschema Handler
  "A handler is a side-effecting computation of actions."
  (s/=> Nil Action))

(s/defschema Middleware
  "Tokamak middleware enable transformations of the 'apply' step of reactor
   operation. "
  (s/=> Handler Handler))

(s/defschema MiddlewareAPI
  "A middleware specification is a means of producing a middleware *given* an
   API against the reactor."
  {:dispatch (s/=> Nil Action)
   :view     (s/=> s/Any (s/protocol p/IView))})