(ns reify.tokamak.reducer
  "A Tokamak reducer is, semantically, a transition function from state to
   state mediated by Tokamak actions. This module contains functions for working
   with reducers and their optimized map version."
  (:refer-clojure :exclude [comp commute])
  (:require
    [taoensso.timbre :as timbre :include-macros true]
    [schema.core :as s :include-macros true]
    [reify.tokamak.schemata :as scm]))

(defn commute
  "Given a map of reducers, `{k (=> s_k e s_k)}`, with each reducer
   specialized to its own unique state type at the given key, return a new
   reducer which operates over the whole state `{k s_k}`."
  [reducer-map]
  (fn [a s]
    (persistent!
      (reduce (fn [s [k r]]
                (assoc! s k (r a (get s k))))
              (transient (or s {})) reducer-map))))

(s/defn comp :- scm/Reducer
  "Produce sequential composition of reducers such that (comp r2 r1) applies
   r1 and then r2. Compositions of reducers each recieve all actions, in other
   words `((comp r2 r1) a s)` is the same as `(r2 a (r1 a s))`."
  ([] (fn identity-reducer [_ s] s))
  ([r :- scm/Reducer & rs :- [scm/Reducer]]
    (fn composed-reducer [a s] (reduce (fn [s r] (r a s)) (r a s) rs))))

(s/defn specialize :- scm/Reducer
  "Specialize a reducer to the subtypes of a particular action type vector. For
   instance, `(specialize [:a :b :c] r)` is a reducer which is sensitive to
   actions with types that have a prefix of `[:a :b :c]` alone. It acts as the
   null reducer otherwise.

   The reducer `r` should expect to see actions which have had their type
   vectors trimmed, eliminating the common prefix."
  [prefix :- [s/Keyword], reducer :- scm/Reducer]
  (fn [{:keys [type] :as a} s]
    (loop [ix 0]
      (if-let [desired-key (get prefix ix)]
        (if-let [observed-key (get type ix)]
          (if (= observed-key desired-key)
            (recur (inc ix))
            ;; else the action mismatches and we return the plain state
            s)
          ;; else the action is too broad; this is probably an error! We don't
          ;; expect to see valid type vectors which are prefixes of other type
          ;; vectors - tel
          (do
            (timbre/warnf "Action inferred to have a non-base type. Saw action
                           with type %s but there's a prefix reducer registered
                           at types below %s." type prefix)
            s))
        ;; If there's no more desired keys then we've whittled away the entire
        ;; prefix! We can trim the action type and pass it to the reducer now.
        (reducer (update a :type subvec ix) s)))))

(s/defschema PrefixReducer
  "A prefix reducer is an interpretation of a tree of reducers as a single
   reducer which enables efficient specification of reducers along with fast
   failure in the action of there being no match."
  {s/Keyword (s/either (s/recursive #'PrefixReducer) scm/Reducer)})

(s/defn of-map :- scm/Reducer
  "Create an efficient reducer by specifying prefixes of the action type vector
   which are being targeted."
  [map :- PrefixReducer]
  (fn [{:keys [type] :as a} s]
    (loop [ix 0
           map map
           prefix []]
      (if-let [observed-key (get type ix)]
        (if-let [submap (get map observed-key)]
          (if (fn? submap)
            (submap (update a :type subvec (inc ix)) s)
            (recur (inc ix) submap (conj prefix observed-key)))
          ;; No submap entails that this action is ignored by this reducer!
          s)
        ;; If we run out of keys in the type then this reducer is more
        ;; specific than the action type observed. This is probably an error
        ;; since we don't expect to see valid type vectors which are prefixes
        ;; of other type vectors - tel
        (do
          (timbre/warnf "Action inferred to have a non-base type. Saw action
                         with type %s but there's a prefix reducer registered
                         at types below %s." type prefix)
          s)))))