(ns reify.tokamak.middleware
  "Default Tokamak middleware for customizing dispatch behaviors."
  (:require
    [schema.core :as s :include-macros true]
    [reify.tokamak.schemata :as scm]))

(s/defschema MiddlewareFactory
  (s/=> scm/Middleware scm/MiddlewareAPI))

(s/defn before :- MiddlewareFactory
  "Before middleware allows one to hook in new effect handlers which recieve
   the action prior to continuing down the chain."
  [effect :- scm/Handler]
  (fn [_api]
    (fn [next]
      (fn [action]
        (effect action)
        (next action)))))

(s/defn after :- MiddlewareFactory
  "After middleware allows one to hook in new effect handlers which recieve
   the action after continuing down the chain."
  [effect :- scm/Handler]
  (fn [_api]
    (fn [next]
      (fn [action]
        (next action)
        (effect action)))))

(s/defn defer :- scm/Middleware
  "Defer middleware enables construction of actions which may defer some of
   their logic until just before being consumed. When defer middleware is used
   it should be placed prior to any middleware which inspects or serializes
   actions and before any middleware which expects the standard FSA style
   actions. Now, events may be functions from an extended MiddlewareAPI to
   events (or, again, functions to be handled by the deref middleware again).
   The extended middleware provides options to both `:continue` the delivery of
   some event constructed on the fly or to `:dispatch` it back to the event
   queue."
  [api :- scm/MiddlewareAPI]
  (fn [next]
    (fn [action]
      (if (fn? action)
        (action (assoc api :continue next))
        (next action)))))

(defn flatmapping
  "If an action recieved is a vector instead of an FSA map, each action is
   handled sequentially; the 'flatmap' middleware."
  [{:keys [dispatch]}]
  (fn [next]
    (fn [action]
      (if (vector? action)
        (doseq [subaction action]
          (dispatch subaction))
        (next action)))))

#?(:cljs
   (s/defn realizing :- scm/Middleware
     "Realizing middleware handles actions containing Javascript style A+
      promises. Must be placed before any middleware which expect the payload
      to be a serializable value. When a promise-like payload is encountered
      its promise is chained to deliver a new action to the queue upon
      resolution. If the promise is successful then the action is re-delivered
      with the promise result mixed into the payload payload. If the promise
      errors out then the action is re-delivered with the result as `:error`
      instead.

      For example, a promise-style action might look like

      { :type [:a :b :c]
        :payload { :promise :parameters }
        :promise #promise<> }

      it will first be delivered with the promise removed (for serialization
      purposes and handling pending states)

      { :type [:a :b :c :pending]
        :payload { :promise :parameters } }

      If the promise resolves successfully then a new event is dispatched

      { :type [:a :b :c :successful]
        :payload { :promise :parameters }
        :result { :some :result } }

      but if it fails then this event is dispatched instead

      { :type [:a :b :c :failed]
        :error { :some :error } }

      Most of the time the actions are returned to the asynchronous action
      channel after their promises resolve, but if the metadata for the action
      states that the action is `:immediate` then it will be passed to the
      synchronous handler."
     ;; TODO Add configurable retry policy as metadata. Emit "retrying" actions.
     [api :- scm/MiddlewareAPI]
     (fn [next]
       (fn [{:keys [promise] :as action}]
         (if promise
           ;; We'll refer to the meta value to determine whether this is
           ;; candidate for immediate continuation
           (let [continue (if (:immediate (meta action))
                            next
                            (:dispatch api))]
             (.. promise
                 (then (fn [value]
                         (continue
                           (-> action
                               (update :type conj :successful)
                               (assoc :result value)
                               (dissoc :promise)))))
                 (catch (fn [err]
                          (continue
                            (-> action
                                (update :type conj :failing)
                                (assoc :error err)
                                (dissoc :promise))))))

             ;; The event itself is passed, still, but the promise is pulled from
             ;; the payload (for printin purposes), relegated to the meta.
             (next
               (-> action
                   (update :type conj :pending)
                   (dissoc :promise)
                   (vary-meta assoc :promise promise))))

           ;; We can do no more for this one, continue down the chain...
           (next action))))))

#?(:cljs
   (s/defn console-logger :- scm/MiddlewareAPI
     "Console logger middleware emits console log entries for each action which passes
      through it. This is most useful when the actions are serializable and
      therefore it may be better placed toward the end of the middleware chain."
     [_api :- scm/MiddlewareAPI]
     (fn [next]
       (fn [{:keys [type] :as action}]
         (.log js/console "%cAction %s" "font-weight: 800" (str type))
         (.dir js/console (clj->js action))
         (next action)))))

;; schema middleware
;; time travel middleware
;; state enlarging middleware