(ns reify.tokamak.machine
  "A Machine is a static depiction of a finite state machine which will be
   executed in a reactor. Machines can be composed from smaller machines (see
   `IMachine` protocol)."
  (:refer-clojure :exclude [reify])
  (:require
    [schema.core :as s]
    [reify.tokamak.reducer :as redu]
    [reify.tokamak.protocols :as p]))

(deftype Machine [initial-state schema reducer])

(defn make
  "Produce a state machine by describing its state, schema, and transition
   semantics."
  [& {:keys [initial-state schema reducer]
      :or {initial-state nil
           schema s/Any
           reducer (fn [e s] s)}}]
  (Machine. initial-state schema reducer))

(defn reify
  "Convert an implementer of IMachine into an actual Machine value."
  [m]
  (Machine. (p/initial-state m)
            (p/state-schema m)
            (p/reducer m)))

(defn- map-keys
  "Apply a function over the keys of a map."
  [f m]
  (persistent! (reduce (fn [m [k v]] (assoc! m k (f v))) (transient {}) m)))

(extend-protocol p/IMachine

  Machine
  (initial-state [m] (.-initial-state m))
  (state-schema [m] (.-schema m))
  (reducer [m] (.-reducer m))

  #?(:cljs function
     :clj clojure.lang.AFn)
  (initial-state [_] nil)
  (state-schema [_] s/Any)
  (reducer [f] f)

  #?(:cljs PersistentHashMap
     :clj clojure.lang.PersistentHashMap)
  (initial-state [m] (map-keys p/initial-state m))
  (state-schema [m] (map-keys p/state-schema m))
  (reducer [m] (redu/commute (map-keys p/reducer m)))

  #?(:cljs PersistentArrayMap
     :clj clojure.lang.PersistentArrayMap)
  (initial-state [m] (map-keys p/initial-state m))
  (state-schema [m] (map-keys p/state-schema m))
  (reducer [m] (redu/commute (map-keys p/reducer m))))