(ns reify.tokamak.core
  "Tokamak provides tools for building Reagent frontend applications with
   centralized state stores, single ratoms. To do this it follows a similar
   design philosophy to re-frame <https://github.com/Day8/re-frame> but avoids
   global state and is made amenable to componentized, reloadable
   architectures."
  (:refer-clojure :exclude [binding])
  (:require
    [reify.tokamak.schemata :as scm]
    [reify.tokamak.view :as view]
    [reify.tokamak.machine :as machine]
    [reify.tokamak.reducer :as reducer]
    [reify.tokamak.protocols :as p]

    #?(:cljs [reify.tokamak.impl.context :refer [stash-context]])
    #?(:cljs [reify.tokamak.reactor :as reactor])
    #?(:cljs [reagent.core :as r])
    #?(:cljs [reagent.ratom :as ratom :include-macros true])

    [schema.core :as s :include-macros true]))

(def Action
  "A Tokamak Flux action (or 'event') is a map of a specific form modeled off
  `acdlite/flux-standard-action`.

  The `type` value specifies the type of the action indicating a (subtyping)
  lattice of specific action types where `[]` is the TOP type.

  The `payload` value qualifies the action and should be typed according to
  the particular type of action being handled.

  The `error` value (as a departure from FSA format) exists iff the action
  signifies an error occurred and in that case is some type of error
  descriptor, specific to the type of action.

  This action specification differs again from FSA by not including a `meta`
  value, but this is, of course, provided naturally through Clojure's own
  metadata facilities."
  scm/Action)

(defn machine
  "Produce a state machine by describing its state, schema, and transition
   semantics."
  [& {:keys [initial-state schema reducer]
      :or   {initial-state nil
             schema        s/Any
             reducer       (fn [e s] s)}}]
  (machine/make :initial-state initial-state
                :schema schema
                :reducer reducer))

#?(:cljs
   (s/defn reactor :- reactor/Reactor
     "Construct a live Reactor from a static Machine. Optionally pass an options
      map which describes a set of middleware to apply (left-to-right), the
      updater function with a signature like `(fn [old-state new-state commit!]
      ...)`, a state ratom to use instead of creating a fresh one, and an event
      channel to use instead of creating a fresh one.

      Generally the reactor behavior is extended using both middlewares and
      custom updater functions---middlewares enable custom logic around the
      interpretation and effect of actions while the updater allows last minute
      state update manipulation like schema validation or history record-keeping."
     ([machine] (reactor/make machine))
     ([machine :- (s/protocol p/IMachine)
       options :- reactor/ReactorOptions]
       (reactor/make machine options))))

#?(:cljs
   (s/defn stop-reactor! :- scm/Nil
     [rx :- reactor/Reactor] (reactor/stop! rx)))

#?(:cljs
   (s/defn dispatch! :- scm/Nil
     "Send an action through a reactor."
     [rx :- reactor/Reactor, a :- scm/Action]
     (reactor/dispatch! rx a)))

#?(:cljs
   (s/defn sub :- scm/Reaction
     "Build a subscription by applying a view to a ratom or reaction. Only
      subscriptions should be bound in a `with-subs` block."
     ([rx :- reactor/Reactor, v :- p/IView] (sub rx v nil))
     ([rx :- reactor/Reactor, v :- p/IView, default :- s/Any]
       (ratom/reaction (view/view v @rx default)))))

#?(:cljs
   (defn render!
     "Starts a normal Reagent renderer with the passed reactor available within
      the DOM tree."
     [rx e comp]
     (r/render [(stash-context {:reactor rx} comp)] e)))

#?(:cljs
   (defn- binding-fn
     "Create a component against the reactor and a set of subscriptions. The
      first argument must by a 'subscription set creator', a function of the
      reactor returning a vector of subscriptions. The signature of the second
      argument must look like `(fn [rx rxns] ...)` where `rxns` is a vector of
      reactions created from each subscription."
     [make-subs render]
     (with-meta
       #(let [cc (r/current-component)
              ctx (.-context cc)
              rx (aget ctx "reactor")
              subs (make-subs rx)]
         (aset cc "subscriptions" subs)
         (apply render rx subs))
       {:context-types
        #js {:reactor (fn prop-type-no-verify
                        [_props _propName _componentName])}

        :component-will-unmount
        (fn destroy-subscriptions [this]
          (let [subscriptions (aget this "subscriptions")]
            (doseq [s subscriptions] (ratom/dispose! s))))})))

#?(:clj
   (defmacro binding
     "Create a *provider* component which binds the ambient reactor and/or
      subscriptions atop it. Subscriptions are automatically destroyed on
      unmount. Since this macro merely *creates* a component it must be *used*
      as normal by placing it in vector brackets.

          [(binding [rx
                     count (sub rx [:count])]
             [:h1 @count])]

      The bindings vector can contain either an even or an odd number of
      forms. If even, each pair of forms is assumed to be a let-binding for a
      subscription (most likely created using `sub`). If odd, the first element
      of the list must be a sym where the

      The body of this macro form *must* return a single component."
     [bindings & body]
     (let [[rx-symbol & sub-bindings] (if (odd? (count bindings))
                                        bindings
                                        (cons (gensym "reactor") bindings))
           sub-names (take-nth 2 sub-bindings)
           sub-forms (take-nth 2 (rest sub-bindings))

           fn-binding-form (vec (cons rx-symbol sub-names))

           make-subs `(fn ~'build-subscriptions [~rx-symbol]
                        (let ~(vec sub-bindings)
                          ~(vec sub-names)))
           render `(fn ~'render-binding
                     ~fn-binding-form
                     ~@body)]
       `(reify.tokamak.core/binding-fn
          ~make-subs
          ~render))))