(ns reify.tokamak.clock
  (:require [reagent.core :as r]
            [reagent.ratom :as ratom]))

(deftype Clock
  [value state running]

  ratom/IDisposable
  (dispose! [_] (reset! running false))

  IAtom
  ratom/IReactiveAtom

  IPrintWithWriter
  (-pr-writer [this writer opts]
    (-write writer "#<Tokamak.Clock: ")
    (pr-writer (goog/getUid this) writer opts)
    (-write writer ">"))

  IDeref
  (-deref [_] (-deref value))

  IWatchable
  (-notify-watches [_ oldval newval] (-notify-watches value oldval newval))
  (-add-watch [_ key f] (-add-watch value key f))
  (-remove-watch [_ key] (-remove-watch value key))

  IHash
  (-hash [this] (goog/getUid this)))

(defn clock
  "Create and initialize a new 'clock' signal. `(clock s f)` is a reaction
   which updates periodically according to the values `[out delay new-s]`
   obtained by applying `(f s)`. Such a clock will immediately take the value
   `out` for the next `delay` milliseconds before changing to the behavior of
   `(clock new-s f)`. If `delay` is `nil` then the clock never updates again."
  [s f]

  (let [[out0 delay0 s0] (f s)
        this (Clock. (r/atom out0) (atom s0) (atom true))
        update-fn (fn update-fn []
                    (let [[out delay s] (f @(.-state this))]
                      (reset! (.-value this) out)
                      (reset! (.-state this) s)
                      (when (and @(.-running this) delay)
                        (js/setTimeout update-fn delay))))]
    (js/setTimeout update-fn delay0)
    this))
