(ns boot.new.recursion-lib
  (:require [clojure.java.shell :as sh]
            [clojure.string :as str]

            [boot.new.templates :as templates]))

(def render (templates/renderer "recursion-lib"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Formatting Helpers

(defn wrap-indent [wrap n list]
  (fn []
    (->> list
         (map #(str "\n" (apply str (repeat n " ")) (wrap %)))
         (str/join ""))))

(defn dep-list [n deps]
  (wrap-indent pr-str n deps))

(defn indent [n list]
  (wrap-indent identity n list))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Template Options

(defn opt-fn-name [opt]
  (symbol (str opt \?)))

(defn opt-id [opt]
  (str \+ opt))

(defmacro defopts [& opt-names]
  `(do ~@(for [opt opt-names]
           `(defn ~(opt-fn-name opt) [opts#]
              (some #{~(opt-id opt)} opts#)))))

(defopts component test-check)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Template Data & Rendering

(defn deps [opts]
  (cond-> []
    (component? opts) (conj '[com.stuartsierra/component "0.3.2" :scope "provided"])))

(defn test-deps [opts]
  (cond-> []
    (test-check? opts) (conj '[org.clojure/test.check "0.9.0" :scope "test"])))

(defn template-data [name opts]
  (let [deps (deps opts)
        test-deps (test-deps opts)]
    {:name name
     :year (templates/year)
     :sanitized (templates/name-to-path name)
     :deps (dep-list 4 deps)
     :deps? (empty? deps)
     :test-deps (dep-list 4 test-deps)
     :test-deps? (empty? test-deps)}))

(defn render-boot-properties []
  (let [{:keys [exit out err]} (sh/sh "boot" "-V" :env {:BOOT_CLOJURE_VERSION "1.9.0-alpha17"})]
    (if (pos? exit)
      (println "WARNING: unable to produce boot.properties file.")
      out)))

(defn mute-implicit-target-warning [boot-props]
  (let [line-sep (System/getProperty "line.separator")]
    (str/join line-sep
                 (conj (str/split boot-props
                                     (re-pattern line-sep))
                       (str "BOOT_EMIT_TARGET=no" line-sep)))))

(defn recursion-lib
  "Generate a minimal library with access to the recursion s3 repository"
  [name & opts]
  (let [data (template-data name opts)]
    (println "Generating fresh 'boot new' recursion-lib project.")
    (apply
      (partial templates/->files data)
      (remove nil?
              [["README.md" (render "README.md" data)]
               (when-let [boot-props (render-boot-properties)]
                 ["boot.properties" (mute-implicit-target-warning boot-props)])
               ["build.boot" (render "build.boot" data)]
               [".gitignore" (render "gitignore" data)]
               [".travis.yml" (render "travis.yml" data)]

               "resources"
               ["src/{{sanitized}}.clj" (render "core.clj" data)]
               ["test/{{sanitized}}_test.clj" (render "core_test.clj" data)]

               ["bin/ci" (render "ci.sh" data)]]))
    (sh/sh "chmod" "+x" (str name "/bin/ci"))))
