(ns boot.new.recursion-app
  (:require [clojure.java.shell :as sh]
            [clojure.string :as str]

            [boot.new.templates :as templates]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Template Options

(defn opt-fn-name [opt]
  (symbol (str opt \?)))

(defn opt-id [opt]
  (str \+ opt))

(defmacro defopts [& opt-names]
  `(do ~@(for [opt opt-names]
           `(defn ~(opt-fn-name opt) [opts#]
              (boolean (some #{~(opt-id opt)} opts#))))
       (defn ~'data-opts [~'opts]
         (into {}
               [~@(for [opt opt-names]
                       (vector (keyword opt)
                               (list (opt-fn-name opt) 'opts)))]))))

(defopts web postgres)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Template Data & Rendering

(def render (templates/renderer "recursion-app"))

(defn template-data [name opts]
  (merge
    {:name name
     :year (templates/year)
     :sanitized (templates/name-to-path name)}
    (data-opts opts)))

(defn render-boot-properties []
  (let [{:keys [exit out err]} (sh/sh "boot" "-V" :env {:BOOT_CLOJURE_VERSION "1.9.0-alpha17"})]
    (if (pos? exit)
      (println "WARNING: unable to produce boot.properties file.")
      out)))

(defn mute-implicit-target-warning [boot-props]
  (let [line-sep (System/getProperty "line.separator")]
    (str/join line-sep
                 (conj (str/split boot-props
                                     (re-pattern line-sep))
                       (str "BOOT_EMIT_TARGET=no" line-sep)))))

(defn recursion-app
  "Generate a minimal library with access to the recursion s3 repository"
  [name & opts]
  (let [data (template-data name opts)]
    (println "Generating fresh 'boot new' recursion-lib project.")
    (apply
      (partial templates/->files data)
      (remove nil?
              [["README.md" (render "README.md" data)]
               (when-let [boot-props (render-boot-properties)]
                 ["boot.properties" (mute-implicit-target-warning boot-props)])
               ["Dockerfile" (render "Dockerfile" data)]
               ["build.boot" (render "build.boot" data)]
               [".gitignore" (render "gitignore" data)]
               [".travis.yml" (render "travis.yml" data)]

               ["resources/logback.xml" (render "logback.xml" data)]
               (when (postgres? opts)
                 "resources/migrations")

               ["config/config-dev.sample.clj" (render "config.clj" data)]
               ["config/config-test.clj" (render "config-test.clj" data)]

               ["dev/logback-test.xml" (render "logback-test.xml" data)]
               ["dev/dev.clj" (render "dev.clj" data)]

               ["src/{{sanitized}}.clj" (render "specs.clj" data)]
               ["src/{{sanitized}}/core.clj" (render "core.clj" data)]
               ["src/{{sanitized}}/utils.clj" (render "utils.clj" data)]
               (when (web? opts)
                 ["src/{{sanitized}}/web.clj" (render "web.clj" data)])

               ["test/{{sanitized}}/core_test.clj" (render "core_test.clj" data)]

               ["deploy.sh" (render "deploy.sh" data)]]))
    (sh/sh "chmod" "+x" (str name "/deploy.sh"))))
