(ns buckshot.scheduler.default
  (:require [buckshot.backend :as backend]
            [buckshot.backend.memory :as mem-backend]
            [buckshot.scheduler :as scheduler]
            [buckshot.util :as util]))

(defrecord DefaultScheduler [active? backend jobs opts]
  scheduler/IScheduler
  (add-job! [_ fn {:keys [id] :as job-opts}]
    (dosync
     (when-not (get @jobs id)
       (when (:period job-opts)
         (alter jobs assoc id [fn job-opts]))
       (let [now (backend/now backend)
             j (util/make-job now fn job-opts)]
         (backend/add-job! backend j)))))
  (del-job! [_ {:keys [id] :as job}]
    (dosync
     (alter jobs dissoc id))
    (backend/del-job! backend job))
  (start! [_]
    (when-not @active?
      (reset! active? true)
      (future (while @active?
                (try (doseq [[_ [fn job-opts]] @jobs
                             :let [now (backend/now backend)
                                   j (util/make-job now fn job-opts)]]
                       (backend/add-job! backend j))
                     (Thread/sleep (:sleep opts))
                     (catch Throwable t
                       (when-let [handler (:exception-handler opts)]
                         (handler t)))))))
    true)
  (stop! [_]
    (reset! active? false)
    true))

(defn make
  "Makes a default Buckshot scheduler.

  'opts' may contain the following:

  :backend - the Buckshot backend to use. Default is the in-memory backend.
  :exception-handler - a function that will be called with any exception
                       thrown by the scheduler.
  :sleep - if specified, the scheduler will sleep for the supplied period,
           i.e. [1 :hour], after scheduling jobs. See buckshot.util/period-fn
           for supported time units (all except :year and :month, which can
           have variable length). Default is 1 second."
  [opts]
  (let [{:keys [backend exception-handler id sleep]} opts
        s (->DefaultScheduler (atom false)
                              (or backend (mem-backend/make))
                              (ref {})
                              {:exception-handler exception-handler
                               :sleep (if sleep
                                        (util/->millis sleep)
                                        1000)})]
    (scheduler/start! s)
    s))
