(ns buckshot.util
  (:require [clj-time.coerce :as time-coerce]
            [clj-time.core :as time]
            [clj-time.periodic :as time-periodic]))

(def period-fns {:year time/years
                 :month time/months
                 :week time/weeks
                 :day time/days
                 :hour time/hours
                 :minute time/minutes
                 :second time/seconds
                 :milli time/millis})

(defn ->period [[x unit]]
  ((get period-fns unit) x))

(defn ->millis [[x unit]]
  (-> [x unit] ->period (.toStandardDuration) (.getMillis)))

(defn make-id []
  (str (java.util.UUID/randomUUID)))

(defn next-start [now start [x unit]]
  (let [s (time-coerce/to-date-time start)
        n (time-coerce/to-date-time now)]
    (->> (time-periodic/periodic-seq s (->period [x unit]))
         (drop-while #(time/before? % n))
         (first))))

(defn make-start [now start period]
  (-> (cond
       (nil? start)  now
       (nil? period) (or start now)
       :else         (next-start now start period))
      (time-coerce/to-long)))

(defn make-job [now fn & [opts]]
  {:args (:args opts)
   :created now
   :fn fn
   :from (.getHostName (java.net.InetAddress/getLocalHost))
   :id (or (:id opts) (make-id))
   :period (:period opts)
   :queue (or (:queue opts) :default)
   :start (make-start now (:start opts) (:period opts))})

(defn stack-trace [t]
  (let [sw (java.io.StringWriter.)]
    (.printStackTrace t (java.io.PrintWriter. sw))
    (.toString sw)))
