(in-ns 'buckshot.backend)
(require '[clojure.walk :as walk])
(require '[taoensso.carmine :as redis])

(def prefix "buckshot:")

(defn- from-redis-key [x]
  (->> x (drop (count prefix)) (apply str) keyword))

(defn- to-redis-key [x]
  (->> x name (str prefix)))

(defn- txn-success? [results]
  (and (seq results)
       (every? #{1} results)))

(defrecord RedisBackend [conn listener]
  IBackend
  (colls [_]
    (->> (redis/wcar conn
           (redis/keys (str prefix "*")))
         (map from-redis-key)))
  (atomically [_ colls f]
    (-> (redis/wcar conn
          (redis/atomically (map to-redis-key colls) (f)))
        txn-success?))
  (hall [_ hash]
    (redis/wcar conn
      (redis/hvals (to-redis-key hash))))
  (hget [_ hash key]
    (redis/wcar conn
      (redis/hget (to-redis-key hash) key)))
  (hadd! [_ hash key value]
    (redis/hset (to-redis-key hash) key value))
  (hrem! [_ hash key]
    (redis/hdel (to-redis-key hash) key))
  (zall [_ set]
    (redis/wcar conn
      (redis/zrange (to-redis-key set) 0 -1)))
  (zmin [_ set max-key]
    (-> (redis/wcar conn
          (redis/zrangebyscore (to-redis-key set) "-inf" max-key "LIMIT" 0 1))
        first))
  (zexists? [_ set value]
    (redis/wcar conn
      (redis/zscore (to-redis-key set) value)))
  (zadd! [_ set key value]
    (redis/zadd (to-redis-key set) key value))
  (zrem! [_ set value]
    (redis/zrem (to-redis-key set) value))
  (publish! [_ channel message]
    (redis/wcar conn
      (redis/publish (name channel) message)))
  (subscribe! [_ channel f]
    (swap! (:state listener) assoc (name channel) (fn [[type ch msg]]
                                                    (when (= type "message")
                                                      (f msg))))
    (redis/with-open-listener listener
      (redis/subscribe (name channel)))))

(defn make-redis [& [spec]]
  (let [conn {:pool {:max-active -1 :max-idle -1}
              :spec spec}
        listener (redis/with-new-pubsub-listener spec {})]
    (RedisBackend. conn listener)))
