(in-ns 'buckshot.backend)
(require '[clojure.walk :as walk])
(require '[taoensso.carmine :as redis])

(defn- txn-success? [results]
  (and (seq results)
       (every? #{1} results)))

(defrecord RedisBackend [conn listener]
  IBackend
  (colls [_]
    (->> (redis/wcar conn
           (redis/keys "*"))
         (map keyword)))
  (atomically [_ colls f]
    (-> (redis/wcar conn
          (redis/atomically (map name colls) (f)))
        txn-success?))
  (hall [_ hash]
    (redis/wcar conn
      (redis/hvals (name hash))))
  (hget [_ hash key]
    (redis/wcar conn
      (redis/hget (name hash) key)))
  (hadd! [_ hash key value]
    (redis/hset (name hash) key value))
  (hrem! [_ hash key]
    (redis/hdel (name hash) key))
  (zall [_ set]
    (redis/wcar conn
      (redis/zrange (name set) 0 -1)))
  (zmin [_ set max-key]
    (-> (redis/wcar conn
          (redis/zrangebyscore (name set) "-inf" max-key "LIMIT" 0 1))
        first))
  (zexists? [_ set value]
    (redis/wcar conn
      (redis/zscore (name set) value)))
  (zadd! [_ set key value]
    (redis/zadd (name set) key value))
  (zrem! [_ set value]
    (redis/zrem (name set) value))
  (publish! [_ channel message]
    (redis/wcar conn
      (redis/publish (name channel) message)))
  (subscribe! [_ channel f]
    (swap! (:state listener) assoc (name channel) (fn [[type ch msg]]
                                                    (when (= type "message")
                                                      (f msg))))
    (redis/with-open-listener listener
      (redis/subscribe (name channel)))))

(defn make-redis [params]
  (let [conn {:pool {:max-active -1 :max-idle -1}
              :spec params}
        listener (redis/with-new-pubsub-listener params {})]
    (RedisBackend. conn listener)))
