(ns buckshot.worker
  (:require [buckshot.core :as buckshot]
            [buckshot.queue :as queue]
            [buckshot.util :as util]
            [clojure.stacktrace :as st]))

(defprotocol IWorker
  (start! [this])
  (stop! [this]))

(defn- do-job [{:keys [queue] :as worker} job]
  (buckshot/log! queue {:worker (:id worker)
                        :type "start"
                        :job job})
  (let [f (get (:fns worker) (:fn job))
        start (util/now-ms)
        result (apply f (:args job))
        end (util/now-ms)]
    (queue/publish! queue (:id job) result)
    (queue/finish-job! queue job)
    (buckshot/log! queue {:worker (:id worker)
                          :type "finish"
                          :job job
                          :duration (- end start)})))

(defrecord Worker [id error-handler fns queue sleep active?]
  IWorker
  (start! [this]
    (when-not @active?
      (reset! active? true)
      (future
        (while @active?
          (try (if-let [j (queue/next-job queue (keys fns))]
                 (if (queue/take-job! queue j)
                   (try (do-job this j)
                        (catch Throwable t
                          (when error-handler
                            (error-handler t)))
                        (finally (Thread/sleep 500)))
                   (Thread/sleep (rand-int 1000)))
                 (Thread/sleep sleep))
               (catch Throwable t
                 (let [trace (with-out-str (st/print-stack-trace t))]
                   (buckshot/log! queue {:worker id
                                         :type "exception"
                                         :trace trace}))))))))
  (stop! [_]
    (reset! active? false)))

(defn make [{:keys [id error-handler fns queue sleep] :as params}]
  (let [w (Worker. id error-handler fns queue (or sleep 500) (atom false))]
    (start! w)
    w))
