(in-ns 'buckshot.backend)
(require '[taoensso.carmine :as redis])

(def pool (redis/make-conn-pool :max-active -1
                                :max-idle -1))

(defmacro with-conn [& body]
  `(redis/with-conn pool ~'spec
     ~@body))

(defn- txn-success? [results]
  (and (seq results)
       (every? #{1} results)))

(defrecord RedisBackend [spec listener]
  IBackend
  (colls [_]
    (->> (with-conn
           (redis/keys "*"))
         (map keyword)))
  (atomically [_ colls f]
    (-> (with-conn
          (redis/atomically (map name colls) (f)))
        txn-success?))
  (del! [_ coll]
    (redis/del (name coll)))
  (hall [_ hash]
    (with-conn
      (redis/hvals (name hash))))
  (hexists? [_ hash key]
    (-> (with-conn
          (redis/hexists (name hash) key))
        pos?))
  (hadd! [_ hash key value]
    (redis/hsetnx (name hash) key value))
  (hrem! [_ hash key]
    (redis/hdel (name hash) key))
  (zall [_ set]
    (with-conn
      (redis/zrange (name set) 0 -1)))
  (zexists? [_ set value]
    (with-conn
      (redis/zscore (name set) value)))
  (zmin [_ set max-key]
    (-> (with-conn
          (redis/zrangebyscore (name set) "-inf" max-key "LIMIT" 0 1))
        first))
  (zadd! [_ set key value]
    (redis/zadd (name set) key value))
  (zrem! [_ set value]
    (redis/zrem (name set) value))
  (publish! [_ channel message]
    (with-conn
      (redis/publish (name channel) message)))
  (subscribe! [_ channel f]
    (swap! (:state listener) assoc (name channel) (fn [[type ch msg]]
                                                    (when (= type "message")
                                                      (f msg))))
    (redis/with-open-listener listener
      (redis/subscribe (name channel)))))

(defn make-redis [params]
  (let [spec (merge (redis/make-conn-spec) params)
        listener (redis/with-new-pubsub-listener spec {})]
    (RedisBackend. spec listener)))
