(ns buckshot.worker
  (:require [buckshot.core :as buckshot]
            [buckshot.queue :as queue]
            [buckshot.util :as util]
            [clojure.stacktrace :as st]))

(defprotocol IWorker
  (start! [this])
  (stop! [this]))

(defn- do-job [{:keys [queue] :as worker} job]
  (buckshot/log! queue {:worker (:id worker)
                        :type "started job"
                        :job job})
  (let [f (get (:fns worker) (:fn job))
        start (util/now-ms)
        result (f worker job)
        end (util/now-ms)]
    (queue/publish! queue (:id job) result)
    (queue/finish-job! queue job)
    (buckshot/log! queue {:worker (:id worker)
                          :type "finished job"
                          :job job
                          :duration (- end start)})))

(defrecord Worker [id fns queue sleep active?]
  IWorker
  (start! [this]
    (when-not @active?
      (reset! active? true)
      (future
        (while @active?
          (try (if-let [j (queue/next-job queue (keys fns))]
                 (if (queue/take-job! queue j)
                   (do (do-job this j)
                       (Thread/sleep 500))
                   (Thread/sleep (rand-int 1000)))
                 (Thread/sleep sleep))
               (catch Throwable t
                 (let [trace (with-out-str (st/print-stack-trace t))]
                   (buckshot/log! queue {:worker id
                                         :type "exception"
                                         :trace trace}))))))))
  (stop! [_]
    (reset! active? false)))

(defn make [{:keys [id fns sleep queue] :as params}]
  (let [w (Worker. id fns queue (or sleep 500) (atom false))]
    (start! w)
    w))
