(in-ns 'buckshot.backend)
(require '[taoensso.carmine :as redis])

(def pool (redis/make-conn-pool :max-active -1
                                :max-idle -1))

(defmacro with-conn [& body]
  `(redis/with-conn pool ~'spec
     ~@body))

(defn- txn-success? [results]
  (and (seq results)
       (every? #{1} results)))

(defrecord RedisBackend [spec listener]
  IBackend
  (get-colls [_]
    (->> (with-conn
           (redis/keys "*"))
         (map keyword)))
  (get-all [_ coll]
    (with-conn
      (redis/zrange (name coll) 0 -1)))
  (get-min [_ coll max]
    (-> (with-conn
          (redis/zrangebyscore (name coll) "-inf" max "LIMIT" 0 1))
        first))
  (has-score? [_ coll score]
    (-> (with-conn
          (redis/zrangebyscore (name coll) score score "LIMIT" 0 1))
        seq))
  (has-item? [_ coll item]
    (with-conn
      (redis/zscore (name coll) item)))
  (atomically [_ f]
    (-> (with-conn
          (redis/atomically [] (f)))
        txn-success?))
  (add! [_ coll score item]
    (redis/zadd (name coll) score item))
  (remove! [_ coll item]
    (redis/zrem (name coll) item))
  (delete! [_ coll]
    (redis/del (name coll)))
  (publish! [_ channel message]
    (with-conn
      (redis/publish (name channel) message)))
  (subscribe! [_ channel f]
    (swap! (:state listener) assoc (name channel) (fn [[type channel message]]
                                                    (when (= type "message")
                                                      (f message))))
    (redis/with-open-listener listener
      (redis/subscribe (name channel)))))

(defn make-redis [params]
  (let [spec (merge (redis/make-conn-spec) params)
        listener (redis/with-new-pubsub-listener spec {})]
    (RedisBackend. spec listener)))
