(ns buckshot.backend.redis
  (:require [buckshot.backend :as bsb]
            [clojure.edn :as edn]
            [taoensso.carmine :as car]))

(def key-prefix "buckshot:")

(def queue-key-prefix "buckshot:queue:")

(defn- ->key [x]
  (str key-prefix (name x)))

(defn- ->queue-key [x]
  (str queue-key-prefix (name x)))

(defn- next-by-queue [queue conn now]
  (first (car/wcar conn
           (car/zrangebyscore (->queue-key queue) "-inf" now "LIMIT" 0 1))))

(defrecord RedisBackend [conn listener listener-agent]
  bsb/IBackend
  (add-job! [_ {:keys [id queue start] :as job}]
    (when (= 1 (car/wcar conn
                 (car/hsetnx (->key :jobs) id job)))
      (car/wcar conn
        (car/zadd (->queue-key queue) start id))
      job))
  (del-job! [_ id queue]
    (let [script "local job = redis.call('hget', 'buckshot:jobs', _:id)
                  if (not job) then
                    return nil
                  else
                    redis.call('hdel', 'buckshot:jobs', _:id)
                    redis.call('zrem', _:qk, _:id)
                    redis.call('zrem', 'buckshot:processing', _:id)
                    return 'ok'
                  end"
          r (car/wcar conn
              (car/lua script {:qk (->queue-key queue)} {:id id}))]
      (= "ok" r)))
  (get-jobs [_ ids]
    (when (seq ids)
      (car/wcar conn
        (apply car/hmget (->key :jobs) ids))))
  (next-job [this queues]
    (let [now (bsb/now this)
          id (some #(next-by-queue % conn now) queues)]
      (car/wcar conn
        (car/hget (->key :jobs) id))))
  (claim-job! [this {:keys [id queue] :as job} worker-id]
    (let [job (assoc job :claimed (bsb/now this) :worker worker-id)
          script "local score = redis.call('zscore', _:qk, _:id)
                  if (not score) then
                    return nil
                  else
                    redis.call('hset', 'buckshot:jobs', _:id, _:job)
                    redis.call('zadd', 'buckshot:processing', score, _:id)
                    redis.call('zrem', _:qk, _:id)
                    return 'ok'
                  end"
          r (car/wcar conn
              (car/lua script {:qk (->queue-key queue)} {:id id :job job}))]
      (when (= "ok" r)
        job)))
  (queues [_]
    (->> (car/wcar conn
           (car/keys (str queue-key-prefix "*")))
         (map #(-> % (subs (count queue-key-prefix)) keyword))))
  (enqueued-ids [_ queues]
    (into {} (for [q queues]
               [q (car/wcar conn
                    (car/zrange (->queue-key q) 0 -1))])))
  (processing-ids [_]
    (car/wcar conn
      (car/zrange (->key :processing) 0 -1)))
  (publish! [_ channel message]
    (car/wcar conn
      (car/publish (->key channel) (pr-str message))))
  (subscribe! [_ channel f]
    (let [k (->key channel)
          msg-handler (fn [[type _ msg]]
                        (when (= type "message")
                          (f (edn/read-string msg))))]
      (send listener-agent (fn [_]
                             (car/with-open-listener listener
                               (car/subscribe k))
                             (swap! (:state listener) assoc k msg-handler)
                             nil))
      true))
  (now [_]
    (let [[seconds micros] (->> (car/wcar conn
                                  (car/time))
                                (map #(Long. %)))]
      (+ (* seconds 1000) (quot micros 1000)))))

(defn make
  "Makes a Redis Buckshot backend. 'opts' may contain :pool and :spec keys,
  in the same format as Carmine connection maps."
  [opts]
  (let [;; supported pool options: https://github.com/ptaoussanis/carmine/blob/master/src/taoensso/carmine/connections.clj
        defaults {:pool {}}
        conn (merge-with merge defaults opts)
        listener (car/with-new-pubsub-listener (:spec conn) {})]
    (->RedisBackend conn listener (agent nil))))
