(ns buckshot.middleware
  (:require [buckshot.backend :as bsb]
            [clojure.stacktrace :as st]))

(defn delete-job
  "Middleware to delete a completed job from the system."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (try
      (handler data)
      (finally
        (bsb/del-job! (:backend worker) (:id job) (:queue job))))))

(defn log!
  "Publishes a log entry on the :log channel via backend pub-sub mechanism."
  [type worker & [extra]]
  (let [b (:backend worker)
        now (bsb/now b)
        log-handler (or (-> worker :opts :log-handler)
                        println)]
    (log-handler (merge {:type type
                         :worker (-> worker :opts :id)
                         :date (pr-str (java.util.Date. now))
                         :timestamp now}
                        extra))))

(defn log-start-finish
  "Middleware to log job start and finish."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (let [b (:backend worker)
          start (bsb/now b)]
      (try
        (log! :start worker {:job job})
        (let [result (handler data)]
          (log! :finish worker {:duration (- (bsb/now b) start)
                                :job job})
          result)
        (catch Throwable t
          (log! :exception worker {:duration (- (bsb/now b) start)
                                   :job job
                                   :trace (with-out-str
                                            (st/print-stack-trace t))})
          (throw t))))))

(defn publish-result
  "Middleware to publish a job result."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (try
      (let [result (handler data)]
        (bsb/publish! (:backend worker) (:id job) result))
      (catch Throwable t
        (bsb/publish! (:backend worker) (:id job) nil)
        (throw t)))))

(defn rethrow-exception
  "Middleware to re-throw if a job returns an exception."
  [handler]
  (fn [data]
    (let [r (handler data)]
      (if (instance? Throwable r)
        (throw r)
        r))))

(defn warn-unknown-fn
  "Middleware to warn about unknown job functions."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (if-not (get (:fn-map worker) (:fn job))
      (throw (Exception. (str "Unknown job fn: " (:fn job))))
      (handler data))))
