(ns buckshot.core
  (:require [buckshot.backend :as bsb]
            [buckshot.backend.memory :as bsbm]
            [buckshot.backend.redis :as bsbr]
            [buckshot.middleware :as bsm]
            [buckshot.scheduler :as bss]
            [buckshot.util :as bsu]
            [buckshot.worker :as bsw]
            [clojurewerkz.quartzite.scheduler :as qs]))

(defn enqueue!
  "Enqueues a job, returns enqueued job id if successful.

  'fn' should be a key in worker function maps.

  'opts' may contain the following:

  :args - seq of arguments for the job function.
  :id - ID for the job. If a job with the same ID is already in the
        system, the new job will not be enqueued. Default is a random UUID.
  :period - if specified, the job will recur with the supplied period, i.e.
            [1 :hour]. See buckshot.util/period-fns for supported time units.
  :queue - queue for the job. Default is :default.
  :start - start time for the job. Must be a Joda DateTime to correctly handle
           Daylight Savings Time. Default is 'now' in UTC."
  [{:keys [backend] :as scheduler} fn & [opts]]
  (let [id (or (:id opts) (bsu/make-id))]
    (when (bss/add-job! scheduler fn (assoc opts :id id))
      id)))

(defn enqueue-p!
  "Enqueues a job, returns enqueued job id and a promise for the result if
  successful. See 'enqueue!' for supported options."
  [{:keys [backend] :as scheduler} fn & [opts]]
  (let [id (or (:id opts) (bsu/make-id))
        p (promise)]
    (bsb/subscribe! backend id #(deliver p %))
    (when-let [id (enqueue! scheduler fn (assoc opts :id id))]
      [id p])))

(defn make-backend
  "Returns new Buckshot backend of specified type. See each type's (make)
  function for supported options."
  [type & [opts]]
  (case type
    :memory (bsbm/make)
    :redis  (bsbr/make opts)))

(defn make-scheduler
  "Returns new Buckshot scheduler.

  'opts' may contain the following:

  :exception-handler - a function that will be called with any exception thrown
                       by the scheduler."
  [backend & [opts]]
  (let [s (bss/map->Scheduler {:active? (atom false)
                               :backend backend
                               :quartz-scheduler (qs/initialize)
                               :opts opts})]
    (bss/start! s)
    s))

(defn make-worker
  "Returns new Buckshot worker.

  'fn-map' should be a map of keys to job functions, which may have arbitrary
  arity. See buckshot.test.core for examples.

  'opts' may contain the following:

  :exception-handler - a function that will be called with any exception thrown
                       by the worker.
  :id - ID for the worker. Default is a random UUID.
  :middleware - seq of worker middleware functions (these are just like Ring
                middleware). Middleware are run in nested order; the last one
                starts first and finishes last. See buckshot.middleware and
                buckshot.test.core for examples.
  :queues - seq of job queues to process. Queues are prioritized by order, with
            the first queue having the highest priority. Default is [:default].
  :sleep - if specified, the worker will sleep for the supplied period, i.e.
           [1 :hour], after processing jobs. See buckshot.util/period-fns for
           supported time units (all except :year and :month, which can have
           variable length). Default is 1 second."
  [backend fn-map & [opts]]
  (let [{:keys [exception-handler id log-handler middleware queues sleep]} opts
        w (bsw/map->Worker {:active? (atom false)
                            :backend backend
                            :fn-map fn-map
                            :thread (atom nil)
                            :opts {:exception-handler exception-handler
                                   :id (or id (bsu/make-id))
                                   :log-handler log-handler
                                   :middleware (concat [bsm/warn-unknown-fn
                                                        bsm/rethrow-exception
                                                        bsm/publish-result
                                                        bsm/delete-job]
                                                       middleware
                                                       [bsm/log-start-finish])
                                   :queues (or queues [:default])
                                   :sleep (if sleep
                                            (bsu/->millis sleep)
                                            1000)}})]
    (bsw/start! w)
    w))
