(ns buckshot.queue
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]
            [clj-time.coerce :as time-coerce]
            [clj-time.core :as time]))

(defprotocol IQueue
  (get-specs [this])
  (get-scheduled [this])
  (get-processing [this])
  (next-job [this fns])
  (add-spec! [this spec])
  (remove-spec! [this spec])
  (schedule-jobs! [this])
  (unschedule-job! [this job])
  (take-job! [this job])
  (finish-job! [this job])
  (publish! [this channel message])
  (subscribe! [this channel f]))

(def period-fns {:year time/years
                 :month time/months
                 :week time/weeks
                 :day time/days
                 :hour time/hours
                 :minute time/minutes
                 :second time/secs
                 :milli time/millis})

(defn- upcoming-starts [t period]
  (let [now (util/now-ms)
        p ((get period-fns period) 1)]
    (->> t
         time-coerce/from-long
         (iterate #(time/plus % p))
         (map time-coerce/to-long)
         (drop-while #(< % now)))))

(defn- upcoming-jobs [specs]
  (let [->jobs (fn [{:keys [start-time period] :as spec}]
                 (for [start (take 3 (upcoming-starts start-time period))]
                   (assoc spec :start-time start)))]
    (->> specs
         (filter :period)
         (mapcat ->jobs))))

(defmacro atomically [& body]
  `(backend/atomically ~'backend
                       (fn [] ~@body)))

(defrecord Queue [backend]
  IQueue
  (get-specs [_]
    (backend/get-all backend :specs))
  (get-scheduled [_]
    (->> (backend/get-colls backend)
         (remove #{:specs :processing})
         (mapcat #(backend/get-all backend %))))
  (get-processing [_]
    (backend/get-all backend :processing))
  (next-job [_ fns]
    (some #(backend/get-min backend %) fns))
  (add-spec! [_ {:keys [id start-time fn] :as spec}]
    (let [now (util/now-ms)
          spec (assoc spec :start-time (or start-time now))]
      (atomically
       (when (and (-> spec :start-time (>= now))
                  (not (backend/has-score? backend :specs (hash id))))
         (backend/add! backend :specs (hash id) spec)
         (backend/add! backend fn (:start-time spec) spec)))))
  (remove-spec! [_ {:keys [fn] :as spec}]
    (atomically
     (backend/remove! backend :specs spec)
     (backend/delete! backend fn)))
  (schedule-jobs! [this]
    (atomically
     (doseq [job (upcoming-jobs (get-specs this))
             :let [{:keys [start-time fn]} job]
             :when (not (backend/has-item? backend :processing job))]
       (backend/add! backend fn start-time job))))
  (unschedule-job! [_ {:keys [fn] :as job}]
    (atomically
     (backend/remove! backend fn job)))
  (take-job! [_ {:keys [start-time fn] :as job}]
    (atomically
     (when (backend/has-item? backend fn job)
       (backend/remove! backend fn job)
       (backend/add! backend :processing start-time job))))
  (finish-job! [_ {:keys [id period] :as job}]
    (atomically
     (when-not period
       (backend/remove! backend :specs job))
     (backend/remove! backend :processing job)))
  (publish! [_ channel message]
    (backend/publish! backend channel message))
  (subscribe! [_ channel f]
    (backend/subscribe! backend channel f)))

(defn make [params]
  (Queue. (:backend params)))
